<?php
//API.class.php
/**
 * Holds the geoAPI class.
 * 
 * @package System
 * @since Version 4.0.0
 */
/**************************************************************************
Geodesic Classifieds & Auctions Platform 5.2
Copyright (c) 2001-2011 Geodesic Solutions, LLC
All rights reserved
http://geodesicsolutions.com
see license attached to distribution
**************************************************************************/
##########SVN Build Data##########
##                              ##
## This File's Revision:        ##
##  $Rev:: 21407              $ ##
## File last change date:       ##
##  $Date:: 2011-05-18 17:59:#$ ##
##                              ##
##################################
/**
 * Requires the XMLRPC PHP Library class, as geoAPI extends IXR_Server in order
 * to communicate.
 */
require_once CLASSES_DIR . 'rpc/XMLRPC.class.php';

/**
 * The main system class for receiving and handling remote API calls, this acts
 * as a translation layer between the communication with the "outside" and each
 * API call.
 * 
 * @package System
 * @since Version 4.0.0
 */
class geoAPI extends IXR_Server {
	/**
	 * Database object, can be used by api tasks by using $this->db
	 *
	 * @var DataAccess
	 */
	public $db;
	
	/**
	 * Session object, can be used by api tasks by using $this->session
	 *
	 * @var geoSession
	 */
	public $session;
	
	/**
	 * geoPC object, can be used by api tasks by using $this->product_configuration
	 *
	 * @var geoPC
	 */
	public $product_configuration;
	
	/**
	 * geoAddon object, can be used by cron tasks by using $this->addon
	 *
	 * @var geoAddon
	 */
	public $addon;
	
	/**
	 * An array of API calls mapped to the absolute file location used for that call.
	 * @var array
	 */
	private $_callbacks;
	
	/**
	 * Constructor, initializes all the class vars and loads the tasks.
	 *
	 * @return geoAPI
	 */
	public function __construct(){
		$this->db = DataAccess::getInstance();
		$this->session = geoSession::getInstance();
		$this->product_configuration = geoPC::getInstance();
		$this->addon = geoAddon::getInstance();
		$this->load();
	}
	
	/**
	 * Loads all the remote API tasks found in the given directory.
	 *
	 * @param string $dir The absolute directory to be looking in.
	 * @param string $methodname The remote API call name, usually just leave this
	 *  at default to load all.
	 */
	public function load ($dir = '', $methodname = 'core')
	{
		if (strlen($dir) == 0){
			if (is_array($this->callbacks) && count($this->callbacks) > 0){
				//already loaded
				return ;
			}
			//loading from base dir
			$dir = API_DIR;
			
			//load addon api's
			$api_addons = $this->addon->getApiAddons();
			
			foreach ($api_addons as $name){
				//go through each api and add it if it is valid.
				if (is_dir(ADDON_DIR.$name.'/api')) {
					$this->load(ADDON_DIR.$name.'/api/','addon.'.$name);
				}
			}
		}
		if (strlen($methodname) == 0){
			$methodname = 'core'; //method should start with the type.
		}
		$dirname = $dir;
		//echo 'Adding dir: '.$dirname.'<br />';
		$dir = opendir($dir);
		while ($filename = readdir($dir)){
			if ($filename !='.' && $filename != '..' && $filename != '_samples' && is_dir($dirname.$filename)){
				//directory, so recursively load
				//echo 'loading sub-dir: '.$dirname.$filename.'<br />';
				
				$this->load($dirname.$filename.'/',$methodname.'.'.$filename);
			} elseif ($filename !='.' && $filename != '..' && $filename != '_samples' && strpos($filename,'.php') !== false && file_exists($dirname.$filename)){
				//echo '<strong>Adding: '.$methodname.'.'.str_replace('.php','',$filename).'</strong><br />';
				$this->addCallback($methodname.'.'.str_replace('.php','',$filename),$dirname.$filename);
			}
		}
		closedir($dir);
	}
	
	/**
	 * Adds a callback (AKA a remote API call) that could be called.
	 * 
	 * Normally this is not called directly.
	 * 
	 * @param string $methodname
	 * @param string $filename
	 */
	public function addCallback($methodname, $filename){
		if (file_exists($filename)){
			$this->callbacks[strtolower($methodname)] = $filename;
		}
	}
	
	/**
	 * Gets the array of valid callbacks.
	 *
	 * @return array
	 */
	public function getCallBacks(){
		$this->load();
		return $this->callbacks;
	}
	
	/**
	 * Runs the specified remote api task, if that task is valid.
	 * 
	 * Usually just used internally.
	 *
	 * @param string $methodname
	 * @param mixed $args
	 */
	public function call ($methodname, $args)
	{
        if (!defined('IN_GEO_API')){
        	define('IN_GEO_API',1);
        }
        $methodname = strtolower($methodname);
        if (count($args) == 1 && is_array($args[0])){
        	$args = $args[0];
        }
		//make sure the security key matches
		if (isset($args['api_key'])){
			//Key passed using associative array
			$passed_key = $args['api_key'];
			unset($args['api_key']);
		} else {
        	//not passed using associative array, see if it is the first item on the array
			$passed_key = array_shift($args);
		}
        
        $site_key = $this->getKeyFor();
        $call_key = $this->getKeyFor($methodname);
        if ($site_key != $passed_key && $call_key != $passed_key){
        	//does not match main site key, see if it matches key specifically for this
        	//api call.
        	return new IXR_Error(-32601, 'Doh! Server error. API key [\'site_key\'] not valid, or not sent.');
        }
        
        //only return the error of the requested method does not exist if the API site key matches up first.
        if (!$this->hasMethod($methodname)) {
            return new IXR_Error(-32601, 'Doh! Server error. Requested method '.$methodname.' does not exist.');
        }
        
        
        $method = $this->callbacks[$methodname];
        // Perform the callback and send the response
        if (is_array($args) && count($args) == 1 && isset($args[0])) {
            // If only one paramater just send that instead of the whole array
            $args = $args[0];
        }
        
        //file should return result.  result can be any php value, or even an IXR_Error if
        //you want to be official when returning an error.
        ob_start();
        $result = require ($method);
        //do nothing with the output.  only consider the return value.
        ob_end_clean();
        return $result;
	}
	
	/**
	 * Gets the site key for the given api call.
	 * 
	 * If no call is given, returns key that works with all api calls.
	 *
	 * Note: API system does needed checks, you should not need to do any checks
	 * within any api call to make sure the key is valid.
	 *
	 * @param string $apiCall
	 * @return string The key for the given remote API call.
	 */
	public function getKeyFor($apiCall=''){
		$site_key = $this->db->get_site_setting('api_key');
        if (strlen($site_key)==0){
        	//no key yet!
        	$site_key = $this->resetKey();
        }
        
        //site key is combination of stored site key, and api call, hashed
        $site_key = sha1($site_key.':_:'.strtolower($apiCall));
        
        return $site_key;
	}
	
	/**
	 * Resets the remote API security key.
	 *
	 * @param string $newKey if blank, a random key is generated.
	 */
	public function resetKey($newKey = ''){
		if (strlen(trim($newKey)) == 0){
			//generate random key
			$to = rand(30,45); //num chars is random, between 30 and 45		
			// define possible characters
			$possible = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-."; 
			for ($i = 0; $i < $to; $i++){
				$newKey .= substr($possible, mt_rand(0, strlen($possible)-1), 1);
			}
		}
		$this->db->set_site_setting('api_key',$newKey);
		return $newKey;
	}
	
	/**
	 * Verifies that the given user token is valid.
	 *
	 * @param string $username
	 * @param string $token
	 * @return boolean
	 */
	public function checkUserToken ($username, $token)
	{
		if (strlen(trim($token)) != 40 || strlen(trim($username)) == 0){
			//token is incorrect
			return false;
		}
		//check the token
		$sql = 'SELECT `api_token` FROM `geodesic_logins` WHERE `username` = ? AND `id` != 1';
		$result = $this->db->Execute($sql, array($username));
		if (!$result || $result->RecordCount() != 1){
			return false;
		}
		$row = $result->FetchRow();
		if ($row['api_token'] === $token){
			return true;
		}
		if (strlen(trim($row['api_token'])) == 0){
			//no user token set, reset token
			$this->resetUserToken($username);
		}
		return false;
	}
	
	/**
	 * Resets a user's remote api token (or sets it for the first time)
	 *
	 * @param string $username
	 * @return Mixed The new api token if successful, or false on failure.
	 */
	public function resetUserToken ($username)
	{
		if (strlen(trim($username)) == 0) {
			//token is incorrect
			return false;
		}
		
		$sql = 'SELECT `id` FROM `geodesic_logins` WHERE `username` = ? AND `id` != 1';
		$result = $this->db->Execute($sql, array($username));
		if (!$result || $result->RecordCount() != 1){
			return false;
		}
		$row = $result->FetchRow();
		$id = $row['id'];
		$key = '';
		do {
			//generate random key
			$key = sha1($username.'abc1'.rand().'23 salt');
			
			$sql = 'SELECT `api_token` FROM `geodesic_logins` WHERE `api_token` = ?';
			$result = $this->db->Execute($sql, array($key));
		} while ($result && $result->RecordCount() > 0);
		
		$sql = 'UPDATE `geodesic_logins` SET `api_token` = ? WHERE `id` = ? LIMIT 1';
		$result = $this->db->Execute($sql, array($key, $id));
		if (!$result) {
			return false;
		}
		return $key;
	}
	
	/**
	 * Function that delays a certain amount of time (5 seconds by default), to be used
	 * when credentials don't match up, so that it artificially makes brute force attacks take 5 seconds
	 * longer per check
	 *
	 * @param string $error_msg
	 * @param int $delay_time The amount of time before showing the error
	 *  message (for security reasons, can artificially increase the amount of
	 *  time it takes, to make it take a lot longer to attempt brute-force)
	 * @return IXR_Error An error object suitable for returning in remote API
	 *  call to signify an error.
	 */
	public function return_error_with_delay ($error_msg, $delay_time = 5)
	{
		sleep($delay_time);
		return new IXR_Error(1000, $error_msg);
	}
	
	/**
	 * Utility function that returns the current time, adjusted 
	 * for the time shift as set in the admin.
	 *
	 * @return int
	 */
	public function time()
	{
		return geoUtil::time();
	}
}