<?php
//Browse.class.php
/**
 * Holds the geoBrowse class.
 * 
 * @package System
 * @since Version 4.0.0
 */
/**************************************************************************
Geodesic Classifieds & Auctions Platform 5.2
Copyright (c) 2001-2011 Geodesic Solutions, LLC
All rights reserved
http://geodesicsolutions.com
see license attached to distribution
**************************************************************************/
##########SVN Build Data##########
##                              ##
## This File's Revision:        ##
##  $Rev:: 20911              $ ##
## File last change date:       ##
##  $Date:: 2011-01-13 15:41:#$ ##
##                              ##
##################################

/**
 * Contains functions common to browsing listings
 * @package System
 * @since Version 4.0.0
 */
class geoBrowse extends geoSite
{
	
	var $configuration_data;
	
	public function __construct()
	{
		parent::__construct();
		
		$this->configuration_data = DataAccess::getInstance()->get_site_settings(true);
		
		if(geoPC::is_ent()) {
			$this->get_category_configuration(0,$this->site_category);
			if ($this->category_configuration['use_site_default']) {
				//NOT using site default (negative logic)
				//there are category-specific settings for this category
				//merge them into the config array as 2nd parameter, so category settings take precedence if they exist
				$this->configuration_data = array_merge($this->configuration_data, $this->category_configuration);
			}
			if (geoPC::is_class_auctions()) {
				//listing type allowed is only setting used whether or not "use site default" is turned on or not
				
				if ($this->category_configuration['listing_types_allowed'] != 0) {
					//using cat-specific setting
					$use = $this->category_configuration['listing_types_allowed'];
				} else {
					$use = $this->configuration_data['listing_type_allowed'];
				}
				
				$this->configuration_data['listing_type_allowed'] = $use;
			}
		}
		
		if ($this->site_category) {
			$category = geoCategory::getBasicInfo($this->site_category);
			if ($category) {
				$view = geoView::getInstance();
				$view->currentCategoryName = $category['category_name'];
				$view->currentCategoryDescription = $category['description'];
			}
		}
	}
	
	/**
	 * converts raw listing data that is common to all browsing pages into a format the templates can use.
	 * This sets up ALL possible data -- it's up to the caller to configure the template to only show the appropriate columns
	 *
	 * @param Array $listing usually a row out of the classifieds table
	 * @param Array $text array of page-specific database text
	 * @param bool $featured is this being displayed as part of a set of featured listings?
	 * 
	 * @return Array the formatted array
	 */
	public function commonBrowseData ($data, $text, $featured=0)
	{
		if (!is_array($data)) {
			//no data given...can't do anything
			trigger_error('DEBUG BROWSE: no data');
			return array();
		}
		
		$formatted = array();
		
		if ($data['item_type'] == 1) {
			$formatted['type'] = $text['item_type']['classified'];
		} else if ($data['item_type'] == 2 || $data['item_type'] == 4) {
			$formatted['type'] = $text['item_type']['auction'];
		}
		
		if ($data['business_type']) {
			$formatted['business_type'] = ($data['business_type'] == 1) ? $text['business_type'][1] : $text['business_type'][2];
		} else {
			$formatted['business_type'] = '';
		}
		
		$no_image_url = ($this->messages[500795])? geoTemplate::getURL('',$this->messages[500795]) : '';
		$photo_icon_url = ($this->messages[500796])? geoTemplate::getURL('',$this->messages[500796]) : '';
		if ($data['image'] > 0) {
			if ($this->configuration_data['photo_or_icon'] == 1) {
				$formatted['full_image_tag'] = true;
				if ($featured) {
					//featured listings have separate size settings for thumbnails
					$width = $this->configuration_data['featured_thumbnail_max_width'];
					$height = $this->configuration_data['featured_thumbnail_max_height'];
				} else {
					//let it use defaults
					$width = $height = 0;
				}
				$formatted['image'] = geoImage::display_thumbnail($data['id'], $width, $height, 1);
			} else {
				$formatted['full_image_tag'] = false;
				$formatted['image'] = $photo_icon_url;
			}
		} else if ($no_image_url) {
			$formatted['full_image_tag'] = false;
			$formatted['image'] = $no_image_url;
		} else {
			$formatted['full_image_tag'] = true;
			$formatted['image'] = '';
		}
		
		$formatted['icons'] = array(
					'sold' => (($data['sold_displayed']) ? true : false),
					'buy_now' => (($data['buy_now'] != 0 && ($data['current_bid'] == 0 || ($this->configuration_data['buy_now_reserve'] && $data['current_bid'] < $data['reserve_price']))) ? true : false),
					'reserve_met' => (($data['reserve_price'] > 0 && $data['current_bid'] > 0 && ($data['item_type'] == 2 && $data['current_bid'] >= $data['reserve_price']) || ($data['item_type'] == 4 && $data['current_bid'] <= $data['reserve_price'])) ? true : false),
					'no_reserve' => ((($data['item_type'] == 2 || $data['item_type'] == 4) && $data['reserve_price'] == 0 && $data['buy_now_only'] == 0) ? true : false),
					'attention_getter' => (($data['attention_getter']) ? true : false),
		);
		$formatted['attention_getter_url'] = geoString::fromDB($data['attention_getter_url']);

		$formatted['title'] = geoString::fromDB($data['title']);

		$description = $data['description'];
		$description = geoFilter::listingDescription($description); //remove bad tags/words
		if($this->configuration_data['display_all_of_description'] != 1) {
			$description = geoFilter::listingShortenDescription($description, $this->configuration_data['length_of_description']); //shorten	
		}
		$formatted['description'] = $description;
		
		$formatted['tags'] = geoListing::getTags($data['id']);
		
		$formatted['address'] = geoString::fromDB($data['location_address']);
		$formatted['city'] = geoString::fromDB($data['location_city']);
		$formatted['state'] = geoString::fromDB($data['location_state']);
		$formatted['country'] = geoString::fromDB($data['location_country']);
		$formatted['zip'] = geoString::fromDB($data['location_zip']);
		
		if ($data['item_type'] == 1) {
			//this is a classified -- show the price
			$price = $data['price'];
		} else if ($data['item_type'] == 2) {
			//this is an auction -- figure out which price to show
			if ($data['buy_now_only'] == 1) {
				//buy now only -- show buy now price
				$price = $data['buy_now'];
			} else if ($data['minimum_bid'] != 0) {
				//minimum bid exists -- show it if it is at least the starting bid
				if ($data['minimum_bid'] < $data['starting_bid']) {
					$data['minimum_bid'] = $data['starting_bid'];
				}
				$price = $data['minimum_bid'];
			} else {
				//show starting bid
				$price = $data['starting_bid'];
			}
		} elseif ($data['item_type'] == 4) {
			//for reverse auctions, the price is always the "minimum_bid" field (which is actually the maximum bid)
			$price = $data['minimum_bid'];
		}
		$formatted['precurrency'] = geoString::fromDB($data['precurrency']);
		$formatted['postcurrency'] = geoString::fromDB($data['postcurrency']);
		$formatted['price'] = geoString::displayPrice($price, $formatted['precurrency'], $formatted['postcurrency']);
		
		for ($i=1; $i <= 20; $i++) {
			$formatted['optionals'][$i] = geoString::fromDB($data['optional_field_'.$i]);
			$field = 'optional_field_'.$i;
			if ($this->fields->$field->field_type=='cost') {
				//display price for any optional fields that "adds cost"
				$formatted['optionals'][$i] = geoString::displayPrice($formatted['optionals'][$i],$formatted['precurrency'], $formatted['postcurrency']);
			}
		}
		
		$formatted['num_bids'] = geoListing::getBids($data['id']);
		
		if ($data['item_type'] == 1 && !$this->fields->classified_start->is_enabled) {
			//this is a classified, and we're not showing classified entry date
			$formatted['entry_date'] = false;
		} else if (($data['item_type'] == 2 || $data['item_type'] == 4) && !$this->fields->auction_start->is_enabled) {
			//this is an auction, and we're not showing auction entry date
			$formatted['entry_date'] = false;
		} else {
			$formatted['entry_date'] = date($this->configuration_data['entry_date_configuration'], $data['date']);
		}

		
		if ($data['item_type'] == 1 && !$this->fields->classified_time_left->is_enabled) {
			//this is a classified, and we're not showing classified time left
			$formatted['time_left'] = false;
		} else if (($data['item_type'] == 2 || $data['item_type'] == 4) && !$this->fields->auction_time_left->is_enabled) {
			//this is an auction, and we're not showing auction time left
			$formatted['time_left'] = false;
		} else {
			$weeks = $this->DateDifference('w',geoUtil::time(),$data['ends']);
			$remaining_weeks = ($weeks * 604800);
			$days = $this->DateDifference('d',(geoUtil::time()+$remaining_weeks),$data['ends']);
			$remaining_days = ($days * 86400);
			$hours = $this->DateDifference('h',(geoUtil::time()+$remaining_days),$data['ends']);
			$remaining_hours = ($hours * 3600);
			$minutes = $this->DateDifference('m',(geoUtil::time()+$remaining_hours),$data['ends']);
			$remaining_minutes = ($minutes * 60);
			$seconds = $this->DateDifference('s',(geoUtil::time()+$remaining_minutes),$data['ends']);

			if ($weeks > 0) {
				$formatted['time_left'] = $weeks.' '.$text['time_left']['weeks'].', '.$days.' '.$text['time_left']['days'];
			} elseif ($days > 0) {
				$formatted['time_left'] = $days.' '.$text['time_left']['days'].', '.$hours.' '.$text['time_left']['hours'];
			} elseif ($hours > 0) {
				$formatted['time_left'] = $hours.' '.$text['time_left']['hours'].', '.$minutes.' '.$text['time_left']['minutes'];
			} elseif ($minutes > 0) {
				$formatted['time_left'] = $minutes.' '.$text['time_left']['minutes'].', '.$seconds.' '.$text['time_left']['seconds'];
			} elseif ($seconds > 0) {
				$formatted['time_left'] = $seconds.' '.$text['time_left']['seconds'];
			} else {
				//listing closed
				$formatted['time_left'] = $text['time_left']['closed'];
			}
		}

		$formatted['edit'] = 'edit';
		$formatted['delete'] = 'delete';
		
		return $formatted;
	}
	
	
	/**
	 * Displays an error message, then exits the script.
	 * If no message is provided, attempts to find one in the site class
	 * Failing that, displays a generic error message
	 * 
	 * @param $error String An error message to show
	 */
	public function browse_error($error='')
	{
		if (!$error && $this->error_message) {
			$error = $this->error_message;
		}
		
		$this->page_id = 1;
		$this->get_text();
		$tpl = new geoTemplate('system','browsing');
		$tpl->error = $error;
		$this->body = $tpl->fetch('error.tpl');
		
		//make it 404 if that setting turned on
		self::pageNotFound();
		
		$this->display_page();
		include_once GEO_BASE_DIR . 'app_bottom.php';
		exit;
	}
	
	/**
	 * If setting to use 404 header is turned on, this method will send a 404
	 * not found header.  Otherwise nothing will happen when this is called.
	 * 
	 * @since Version 5.1.0
	 */
	public static function pageNotFound ()
	{
		if (DataAccess::getInstance()->get_site_setting('use_404')) {
			//use 404 status code, so search engines don't index this page any more.
			header ('HTTP/1.0 404 Not Found', 404);
		}
	}
	
	/**
	 * Returns the order by part of a SQL statement depending on the specified
	 * browse type number, specify false to use site-default.
	 * 
	 * @param int $browse_type The number to browse by.
	 * @return string
	 */
	public function getOrderByString($browse_type=false)
	{
		if ($browse_type === false) {
			//nothing passed in -- check class var
			$browse_type = $this->browse_type;
		}
		$sort_types = array (
			1 => array('minimum_bid','price'),
			3 => 'date',
			5 => 'title',
			7 => 'location_city',
			9 => 'location_state',
			11 => 'location_country',
			13 => 'location_zip',
			15 => 'optional_field_1',
			17 => 'optional_field_2',
			19 => 'optional_field_3',
			21 => 'optional_field_4',
			23 => 'optional_field_5',
			25 => 'optional_field_6',
			27 => 'optional_field_7',
			29 => 'optional_field_8',
			31 => 'optional_field_9',
			33 => 'optional_field_10',
			//dups for city, state, country and zip
			//TODO: remove dups once confirmed that one set or the other is not
			// referenced any more.
			35 => 'location_city',
			37 => 'location_state',
			39 => 'location_country',
			41 => 'location_zip',
			43 => 'business_type',
			45 => 'optional_field_11',
			47 => 'optional_field_12',
			49 => 'optional_field_13',
			51 => 'optional_field_14',
			53 => 'optional_field_15',
			55 => 'optional_field_16',
			57 => 'optional_field_17',
			59 => 'optional_field_18',
			61 => 'optional_field_19',
			63 => 'optional_field_20',
			//65 => '',  ////***65/66 - reserved cases, default for some SEO pages***
			69 => 'ends',
			//copy of #3, for "legacy", this should probably be
			//removed at some point once the use of it has been confirmed to
			//have been removed..
			67 => 'date',
			71 => 'image > 0', //this is valid mysql: "ORDER BY image > 0 DESC" means "show listings with at least one image first"
		);
		
		//ODD IS ASC, EVEN THAT FOLLOWS IS DESC
		//e.g. 61 => optional 19 ASC ~~ 62 => optional 19 DESC
		
		//Sort order by ASC or DESC depend on if number is even or odd.
		//even numbers are DESC, odd numbers are ASC (a%2=0 means even, a%2=1 means odd)
		$asc_desc = ($browse_type % 2 == 0)? 'DESC' : 'ASC';
		
		//fix ones where odd version is desc, and even version is asc (backwards of normal) 
		$asc_backwards = array (
			//if there are ever any where odd num is DESC and even is ASC, add that number to
			//this array, for instance if 1 and 2 were backwards it would look like:
			//1,2
		);
		if (in_array($browse_type,$asc_backwards)) {
			//backwards ones, even # is DESC, odd # is ASC (a%2=0 means even, a%2=1 means odd)
			$asc_desc = ($browse_type % 2 == 0)? 'ASC' : 'DESC';
		}
		
		//Goal: if it's an even number, get it to be 1 less.  (a%2=0 means even, a%2=1 means odd)
		$browse_type = (($browse_type % 2 == 0)? $browse_type - 1: $browse_type);
		
		
		if ($browse_type <= 0 || !isset($sort_types[$browse_type])) {
			//default case, this is a special case where better placement comes first!
			$order_by = "ORDER BY `better_placement` DESC, `date` DESC";
		} else {
			//use some fanciness with arrays to avoid a huge gigantic long switch.
			$sort_fields = (is_array($sort_types[$browse_type]))? $sort_types[$browse_type]: array($sort_types[$browse_type]);
			$sort = array();
			foreach ($sort_fields as $field){
				$sort [] .= "$field $asc_desc"; //don't use `backticks` on $field, or inequality conditions (like 71) will break
			}
			$sort = implode(', ',$sort);
			$order_by = "ORDER BY $sort, `better_placement` DESC";
		}
		//die($order_by);
		return $order_by;
	}
	 
	/**
	 * Finds out if a given id number has a listing associated with it.
	 * Included here mainly for legacy purposes, this may be removed in the
	 * future.  Instead, you would call geoListing::getListing($classified_id,false)
	 * and if the result produced an object, then the listing exists.  If it
	 * returned false, you know the listing does not exist.
	 *
	 * @param int $classified_id the id to check
	 * @return bool true if listing exists, false otherwise
	 */
	public function classified_exists ($classified_id=0)
	{
		if (!is_numeric($classified_id) || $classified_id <= 0) {
			return false;
		}
		$listing = geoListing::getListing($classified_id, false);
		return is_object($listing);
	}
	
	/**
	 * Populates category browsing information, according to $this->site_category,
	 * using category cache if available.  Currently it returns the navigation
	 * contents, but that could change once we ge around to re-doing it so that
	 * the category data is cached using the geoCache system instead of just
	 * saving category browsing to DB.
	 * 
	 * @param array $text Associative array of text to use in template, with following
	 *   indexes: back_to_normal_link, tree_label, main_category, no_subcats
	 * @param string $cacheNamePrefix The DB column prefix for the category cache
	 * @return string The category browsing HTML to use on the page.
	 * @since Version 5.1.0
	 */
	public function categoryBrowsing ($text = array(), $cacheNamePrefix = '')
	{
		if (geoPC::is_print() && $this->db->get_site_setting('disableAllBrowsing')) {
			return;
		}
		$category_cache = false;
		if ($this->site_category && $this->db->get_site_setting('use_category_cache') && !$this->db->isBrowsingWhereClause()) {
			//see if we have a valid cache saved
			$sql = "select `{$cacheNamePrefix}category_cache` from ".geoTables::categories_languages_table." where {$cacheNamePrefix}cache_expire > ? and category_id = ? and language_id = ?";
			$category_cache = geoString::fromDB($this->db->GetOne($sql, array(geoUtil::time(), $this->site_category, $this->language_id)));
		}
		
		if (!$category_cache) {
			//no cache saved -- make a new one
			
			//get the categories inside of this category
			$sql = "SELECT lang.category_id, lang.category_name, lang.description, lang.language_id, lang.`category_cache`, lang.`cache_expire`,
				cat.category_image, cat.auction_category_count, cat.category_count
				FROM ".geoTables::categories_table." as cat, ".geoTables::categories_languages_table." as lang where
				cat.parent_id = ".$this->site_category." and cat.category_id = lang.category_id and lang.language_id = ".$this->language_id." 
				order by cat.display_order, lang.category_name";
				
			$category_result = $this->db->Execute($sql);
	
			if (!$category_result) {
				$this->error_message = "<span class=\"error_message\">".urldecode($this->messages[65])."</span>";
				return false;
			}
			
			//get category list, then optionally save it to cache for next time
			$cacheTpl = new geoTemplate('system','browsing');
			$cacheTpl->assign(geoView::getInstance()->getAllAssignedVars());
			$cacheTpl->category = $this->site_category;
			if (!$text) {
				//set default text
				$text = array(
					'back_to_normal_link' => $this->messages[876],
					'tree_label' => $this->messages[680],
					'main_category' => $this->messages[18],
					'no_subcats' => $this->messages[20]
				);
			}
			$cacheTpl->text = $text;
			
			if(!$this->browse_type) {
				$this->browse_type = 0;
			}
			$a_var = ($_GET['a'])? (int)$_GET['a'] : 5;
			$c_str = ($_GET['c'])? '&amp;c='.(int)$_GET['c'] : ''; //preserve sort order, but default to nothing if not present
			$d_str = ($_GET['d'])? '&amp;d='.(int)$_GET['d'] : ''; //preserve lookback (for "newest listings" pages), default to nothing if not present
			$cacheTpl->link = $this->db->get_site_setting('classifieds_file_name')."?a=$a_var".$c_str.$d_str.$this->browsing_options['query_string']."&amp;b=";
		
			$cacheTpl->tree_display_mode = $this->configuration_data['category_tree_display'];
			if ($this->configuration_data['category_tree_display'] != 3) {
				$category_tree = geoCategory::getTree($this->site_category);
				if(is_array($category_tree)) {
					krsort($category_tree);
					$cacheTpl->array_tree = $category_tree;					
				} else {
					$cacheTpl->string_tree = $category_tree;
				}
					
				$cacheTpl->browse_type = $browse_type;
			}
			
			$current_category_name = geoCategory::getName($this->site_category);
			$cacheTpl->current_category_name = $current_category_name->CATEGORY_NAME;
			if ($this->configuration_data['display_category_navigation'])
			{
				if (!$category_result->RecordCount()) {
					if ($this->configuration_data['display_no_subcategory_message']) {
						$cacheTpl->show_no_subcats = true;
					} else {
						//no subcats to show, but option to show 'no subcats' message is off
					}
				} else {
					$cacheTpl->show_subcats = true;
					$cacheTpl->category_columns = $columns = ($this->site_category) ? $this->configuration_data['number_of_browsing_subcategory_columns'] : $this->configuration_data['number_of_browsing_columns'];
					$cacheTpl->column_width = floor(100 / $columns) . '%';
					
					$categories = array();
					
					$category_new_ad_limit = $this->db->get_site_setting('category_new_ad_limit');
					
					$catCount = 0;
					//simple hack for now, don't show categories unless there is no
					//main category and there is no "back to normal" link.
					$showSubcategories = !($this->site_category) && !$text['back_to_normal_link'];
					
					while ($row = $category_result->FetchRow()) {
						//let category class have data so it doesn't need to look it up again
						//in the same page load
						geoCategory::addCategoryResult($row);
						if ($this->configuration_data['display_category_count']) {
							$row ['category_count']= $this->display_category_count(0, $row['category_id']);
						} else {
							//admin has showing counts turned off, so remove the count data
							$row['category_count'] = false;
						}
						$row ['category_name']= geoString::fromDB($row['category_name']);
						$row ['category_description']= geoString::fromDB($row['description']);
						if ($category_new_ad_limit) {
							$row ['new_ad_icon'] = geoCategory::new_ad_icon_use($row['category_id']);
						}
						$row['count_add'] = 1;
						if ($showSubcategories) {
							$sql = "SELECT lang.category_id, lang.category_name, lang.description, lang.language_id, lang.`category_cache`, lang.`cache_expire`,
								cat.category_image, cat.auction_category_count, cat.category_count FROM ".geoTables::categories_table." as cat, ".geoTables::categories_languages_table." as lang where
								cat.parent_id = ".(int)$row['category_id']." and cat.category_id = lang.category_id and lang.language_id = ".$this->language_id." 
								order by cat.display_order, lang.category_name";
							$row['sub_categories'] = $this->db->GetAll($sql);
							//let the category class have the subcategory data, so it doesn't have
							//to look it up again during this page load
							geoCategory::addCategoryResults($row['sub_categories']);
							
							//2 "sub category" is roughly the height as single main category
							if ($row['sub_categories']) {
								$row['count_add'] = ceil(count($row['sub_categories'])/2) + 1;
							}
						}
						$catCount += $row['count_add'];
						$categories [$row['category_id']] = $row;
					}
					
					$cat_alpha = $this->db->get_site_setting('cat_alpha_across_columns');
					
					
					$maxColumnCount = ceil($catCount/$columns);
					
					$currentColumn = $currentCount = 0;
					
					$categories_sorted = array();
					foreach ($categories as $row) {
						$categories_sorted [$currentColumn][] = $row;
						if ($cat_alpha) {
							$currentColumn++;
							//reset back to 0 if needed.
							if ($currentColumn >= $columns) $currentColumn = 0;
						} else {
							$currentCount += $row['count_add'];
							if ($currentCount >= $maxColumnCount && $currentColumn < $columns) {
								//make column count "start" at where it would start if the
								//previous column was allowed to newline on sub cats
								$currentCount = $currentCount%$maxColumnCount;
								$currentColumn++;
							}
						}
					}
					
					$cacheTpl->categories = $categories_sorted;
					
					$cacheTpl->show_descriptions = $this->db->get_site_setting('display_category_description');
				}
			}
			$category_cache = $cacheTpl->fetch('common/category_block.tpl');
			
			if ($this->site_category && $this->db->get_site_setting('use_category_cache') && !$this->db->isBrowsingWhereClause()) {
				$recache_time = geoUtil::time() + (3600 * $this->db->get_site_setting('use_category_cache'));
				$sql = "update ".$this->db->geoTables->categories_languages_table." set
					{$cacheNamePrefix}category_cache = ?,
					{$cacheNamePrefix}cache_expire = ?
					where category_id = ? and language_id = ?";
				$cache_result = $this->db->Execute($sql, array(geoString::toDB($category_cache), $recache_time, $this->site_category, $this->language_id));
				if(!$cache_result) {
					trigger_error('DEBUG CACHE: failed to save cache');
				}
			}
		}
		return $category_cache;
	}
}