<?php
//Cart.class.php
/**
 * Holds the geoCart class.
 * 
 * @package System
 * @since Version 4.0.0
 */
/**************************************************************************
Geodesic Classifieds & Auctions Platform 5.2
Copyright (c) 2001-2011 Geodesic Solutions, LLC
All rights reserved
http://geodesicsolutions.com
see license attached to distribution
**************************************************************************/
##########SVN Build Data##########
##                              ##
## This File's Revision:        ##
##  $Rev:: 20911              $ ##
## File last change date:       ##
##  $Date:: 2011-01-13 15:41:#$ ##
##                              ##
##################################

require_once(CLASSES_DIR.'site_class.php');

//Temporary site class, to be used to make refactoring the site and sell class easier
require_once(CLASSES_DIR .'order_items/_site_class_temp.php');

//TODO: remove this once we go live, this is just to make sure we don't have any order items with same class as template
require_once(CLASSES_DIR . 'order_items/_template.php');
/**
 * This class is behind the cart, loading all the order items and such and displaying,
 * and processing all the different pages.
 * 
 * @package System
 * @since Version 4.0.0
 */
class geoCart
{
	//Constants used in addStep call.
	const FIRST_STEP = 'first';
	const LAST_STEP = 'last';
	const AFTER_STEP = 'after';
	const BEFORE_STEP = 'before';
	const REPLACE_STEP = 'replace';
	
	/**
	 * This is the "main type" that is currently being "worked on" in the cart.
	 * If not working on any specific item type, will be set to 'cart'.
	 * 
	 * @var string
	 */
	public $main_type;
	
	/**
	 * The DataAccess object for easy access.
	 * @var DataAccess
	 */
	public $db;
	
	/**
	 * The geoSession object for easy access.
	 * @var geoSession
	 */
	public $session;
	
	/**
	 * The cart vars, used to hold vars specific to the current "cart session".
	 * 
	 * @var array
	 */
	public $cart_variables;
	
	/**
	 * The current user's data.
	 * @var array
	 */
	public $user_data;
	
	/**
	 * An array of all the current price plan settings.
	 * @var array
	 */
	public $price_plan;
	
	/**
	 * Associative array of actions and what item types those actions were performed
	 * on if applicable.
	 * 
	 * Useful for telling if a particular item type was just canceled, in order to do
	 * something special at the time the cart is being displayed.
	 * 
	 * @var array
	 */
	public $actions_performed = array();
	/**
	 * Order attached to the cart
	 *
	 * @var geoOrder
	 */
	public $order;
	
	/**
	 * The current step the cart is on.
	 * @var string
	 */
	public $current_step;
	/**
	 * Order item, the type will match main_type.
	 *
	 * @var geoOrderItem
	 */
	public $item;
	
	protected $all_steps = array();
	
	/**
	 * Number of errors currently accumulated, if there are any number more than
	 * 0 the cart will not proceed to the next step automatically.  Don't modify
	 * this directly, instead use {@link geoCart::addError()}
	 * 
	 * @var int
	 */
	public $errors = 0;
	
	/**
	 * An array of error messages.
	 * @var array
	 */
	public $error_msgs = array();
	/**
	 * Site (temporary till we move everything in site to other classes)
	 *
	 * @var tempSiteClass
	 */
	public $site;
	
	private $built_in_steps = array(
		'delete', //delete item, special step
		'preview', //preview item, special step
		'other_details',
		'cart',
		'payment_choices',
		'process_order' //processing is done in payment_choicesProcess...
	);
	private $action = 'cart';
	private $actionSpecial = '';
	private $registry;
	private static $_instance;
	/**
	 * Used internally to remember whether there has been changes to the order since it was last
	 *  serialized.  If there is not changes, when serialize is called, nothing will be done.
	 *
	 * @var boolean
	 */
	private $_pendingChanges;
	
	private $_doProcess, $_skipInitSteps = false;
	
	/**
	 * Whether or not the cart has been "fully" initialized or not.
	 * @var bool
	 */
	public $initialized_full = false;
	
	/**
	 * Whether or not the cart has been partially inited or not.
	 * @var bool
	 */
	public $initialized_onlyItems = false;
	
	/**
	 * Only valid way to get an instance of the geoCart.
	 * 
	 * You would do something like:
	 * 
	 * $cart = geoCart::getInstance();
	 *
	 * @return geoCart Instance of geoCart
	 */
	public static function getInstance()
	{
		if (!(isset(self::$_instance) && is_object(self::$_instance))){
    		$c = __class__;
    		self::$_instance = new $c();
    	}
    	return self::$_instance;
	}
	
	/**
	 * Do not create new cart object, instead use geoCart::getInstance()
	 */
	private function __construct()
	{
		$this->errors = 0;
		//let app_bottom know it needs to save the cart
		define ('geoCart_LOADED',1);
	}
	
	/**
	 * Initializes the cart for the first time in the page load.
	 * 
	 * @param bool $onlyInitItems if true, will only initialize the cart order
	 *  items, and not do any of the other stuff.
	 * @param int $userId If set, will create the cart for the given user instead
	 *  of the user from the current session.
	 * @param bool $renderPage Only used if $onlyInitItems is false, if this is
	 *   true, the page will be displayed, if false, everything will be done
	 *   except any processing or checkvars calls, or the last step of 
	 *   displaying the page.
	 */
	public function init ($onlyInitItems = false, $userId = null, $renderPage = true)
	{
		//TODO: everywhere that returns false in this init, instead display an error or something
		//set up session
		
		trigger_error('DEBUG CART: START');
		
		if ($this->initialized_full) {
			//we've already done a full init
			return;
		} else if($this->initialized_onlyItems && $onlyInitItems) {
			//only want to init items, and have already init'd items
			return;
		}
		
		//let anyone who cares know that we've run init()
		if ($onlyInitItems) {
			$this->initialized_onlyItems = true;
		} else {
			$this->initialized_full = true;
		}
		
		$this->session = geoSession::getInstance();
		
		//init main class vars
		$this->db = DataAccess::getInstance();
		
		//NOTE: to allow anonymous listings, the check to see if a user is logged in
		//is now handled in the initItem process.
		
		//set default, do not checkvars/process
		$this->_doProcess = false;
		
		
		//TODO: Remove this once everything has been moved out of site class!
		$this->site = Singleton::getInstance('tempSiteClass');
		
		$this->site->inAdminCart = (defined('IN_ADMIN'));
		
		if (!$this->_getUserData($userId)){
			return false; //initialize $this->user_data 
		}
		if (!$this->setPricePlan()){
			return false; //initialize $this->price_plan
		}
		
		geoPaymentGateway::setGroup($this->user_data['group_id']); //let payment gateway know which group to get payment gateway settings for
		
		//Initialize session
		if (!$this->initSession(0, $onlyInitItems)) {
			//something went wrong with session
			trigger_error('DEBUG CART: Returning false, init session came back false.');
			return false;
		}
		if ($onlyInitItems) {
			//this is probably some 3rd party or special case, who is only
			//interested in initializing up to the point of getting the items set up.
			
			//oh be sure to populate text though
			$this->site->messages = $this->db->get_text(true);
			
			return true;
		}
		if (isset($_GET['action_special']) && $_GET['action_special'] == 'cancel_and_go') {
			//first, cancel current process, they clicked on the cancel and continue link.
			$this->performAction('cancel');
			$this->action = $this->main_type = 'cart';
			
		}
		//If there is an action, run action
		$actions_outside_cart = array(
			'cancel',
			'process'
		);
		if (isset($_GET['action']) && strlen(trim($_GET['action'])) > 0) {
			if ($this->main_type == 'cart' || (in_array($_GET['action'],$actions_outside_cart))) {
				if ($this->main_type == 'cart' && $_GET['action'] == 'process' && isset($_GET['step']) && !in_array($_GET['step'], array('cart','payment_choices','process_order'))) {
					//It is in a main step of a cart, the action is process, but the step is not one of the steps 
					//you would do here, so they are probably hitting refresh right after approving the last
					//step in adding an item.
					
					//nothing to do in this case.
					
				} else {
					//perform an action, only call if not currently in middle of adding item to cart
					$this->performAction(trim($_GET['action']));
				}
			} else if ($_GET['action'] == 'forcePreview' && $this->cart_variables['order_item'] != -1) {
				//action is preview, and we're in the middle of something...
				//allow previewing no matter where we are in the cart.
				$currentCartVars = $this->cart_variables;
				
				$this->performAction('preview');
				$this->displayStep();
				
				//put it back where it was at before
				$this->cart_variables = $currentCartVars;
				trigger_error('DEBUG CART: END');
				return true;
			} else {
				//user is attempting to start something new, but we are in the middle of something.
				//so show them that message.
				
				return $this->displayCartInterruption();
			}
		}
		if (strlen($this->main_type) == 0) {
			//if main type is not set, set it to "cart"
			$this->main_type = $this->cart_variables['main_type'] = 'cart';
		}
		
		if (!$this->_skipInitSteps) {
			//initialize the steps
			$this->initSteps();
			trigger_error('DEBUG CART: Init steps.  Steps: '.print_r($this->all_steps,1));
		} else {
			trigger_error('DEBUG CART: Init steps SKIPPED!');
		}
		
		//If process is set, and the number of steps between the URL step and the session step
		//is less than 2, and renderPage is true, then process this step.
		if ($this->_doProcess && $renderPage) {
			//Need to do work some work
			//First, Check Vars for this step
			
			if ($this->checkVars()) {
				//check vars was good, now call process
				//note that process will make the current step be incremented by one.
				$this->processStep();
			}
		}
		//Now display, if display affects anything to do with the cart it better save itself...
		if ($renderPage) {
			$this->displayStep();
		} else {
			//call the main cart display, but have it do everything except
			//for actually displaying the page.
			$this->cartDisplay($renderPage);
		}
		
		//NOTE: Cart is saved in app_bottom.php so as long as the application exits properly, the cart session should be saved.
		
		trigger_error('DEBUG CART: END');
	}
	
	/**
	 * Used internally, to initialize the cart session data.
	 * 
	 * @param int $trys Number of times session was inited.
	 * @param bool $restoreOnly If true, will not create a new session, or something
	 *  like that...
	 * @return bool True if successful, false otherwise.
	 */
	protected function initSession ($trys = 0, $restoreOnly = false)
	{
		$allowNew = false;
		if (defined('IN_ADMIN')) {
			$adminId = (int)$this->session->getUserId();
			if (!$adminId) {
				//something wrong, we're in admin but there is no admin ID, this shouldn't happen
				return false;
			}
			$userId = (int)$this->user_data['id'];
			$sessionId = 0;
		} else {
			$adminId = 0;
			$userId = (int)$this->session->getUserId();
			$sessionId = ($userId)? 0: $this->session->getSessionId();
		}
		//let our site class know what the user id is.
		$this->site->userid = $this->site->classified_user_id = $userId;
		
		$sql = "SELECT * FROM ".geoTables::cart." WHERE `user_id` = ? AND admin_id = ? AND `session` = ? ORDER BY `order_item` ASC LIMIT 1";
		$query_data = array($userId, $adminId, $sessionId);
		$result = $this->db->GetRow($sql, $query_data);
		if ($result === false) {
			trigger_error('ERROR ORDER SQL: Sql: '.$sql.' Error Msg: '.$this->db->ErrorMsg());
			trigger_error('DEBUG CART: Return False');
			return false;
		}
		if ($result) {
			//get order and make sure it's valid
			$order = ($result['order'])? geoOrder::getOrder($result['order']): 0;
			if (!is_object($order) || $order->getId() != $result['order']) {
				if ($result['order']) {
					//order id is set, but order is no good.  This could be because the entry in the order table (or
					//even the entire table) has been wiped, or it could be some sort of DB error.
					trigger_error('ERROR CART: Init session, getting order, order is no good! going to start a new cart session and inform the user.');
					//show an error message to the user, if this is being caused by some sort of DB error, we don't want
					//to be silent about it as it would make for a very hard problem to troubleshoot.
					$this->addErrorMsg('generic','Internal Error: Your cart contents have been reset, as we could not retrieve the details of your cart.  If you continually see this error message,
					please inform the site admin.');
					
					//kill the cart from the DB
					$this->db->Execute ("DELETE FROM ".geoTables::cart." WHERE `user_id` = ? AND `admin_id` = ? AND `session` = ? LIMIT 1", $query_data);
				}
				//make it create a new cart
				$result = $order = false;
			}
		}
		
		if ($result) {
			trigger_error('DEBUG CART: Existing cart session, restoring session.');
			//pre-existing session, initialize common vars.
			$this->cart_variables = $result;
			
			$this->order = $order;
			//re-set order buyer, in case switching from anon. to logged in
			$this->order->setBuyer($userId);
			//re-set order admin, in case something weird happened..
			$this->order->setAdmin($adminId);
			
			//set main type
			if (strlen($this->cart_variables['main_type'])){
				//force the main type if set in session, can't be switching around now!
				$this->main_type = $this->cart_variables['main_type'];
			}
			
			//get item
			$this->item = 0;
			if ($this->cart_variables['order_item'] > 0) {
				if (!$this->initItem($this->cart_variables['order_item'],false) || ($this->main_type != 'cart' && $this->item->getType() != $this->main_type)) {
					//what the.. order item is no good?
					trigger_error('DEBUG CART: Item not to be set, item: <pre>'.print_r($this->item,1).'</pre> main type: '.$this->main_type);
					$item = 0;
				}
			} elseif ($this->cart_variables['order_item'] == -1) {
				//this is a stand-alone cart, so the item is going to be the only item in the order.
				$items = $this->order->getItem();
				if (is_array($items)){
					foreach ($items as $item){
						if (is_object($item)){
							//since there may be other stuff in that array, like "sorted", go through each one and the first one that is
							//an object, that must be the only item attached to this order.
							$this->item = $item;
							break;
						}
					}
				}
				if ($trys == 0 && !is_object($this->item)) {
					//could not get item?  something went wrong, kill cart and try to init session again
					
					$this->removeSession();
					return $this->initSession(1, $restoreOnly);
				}
			}
			
			//touch the session, this will be used when session vars are saved..
			$this->cart_variables['last_time'] = geoUtil::time();
			
			$vars = null; //add vars here if needed to be passed
			//let the main type initialize anything else, including
			//making any calls to sub-types or whatever...
			geoOrderItem::callUpdate('geoCart_initSession_update',$vars);
		} else if (!$restoreOnly) {
			//create new
			trigger_error('DEBUG CART: No existing cart session, creating new session.');
			//sell session data not there yet...start over
			
			//create new order for this cart
			//create order object
			$order = new geoOrder();
			
			//set up order's info, if the individual order item wants to change any of these, they can.
			$order->setSeller(0);//seller is 0, it is the site doing the "selling", selling the ability to place the listing.
			$order->setBuyer($userId); //set buyer to be this user
			$order->setAdmin($adminId); //set admin ID on order
			$order->setParent(0);//this is main listing order
			
			$order->setCreated(geoUtil::time());
			
			$this->order = $order;
			
			$this->item = null;
			$this->main_type = 'cart';
			
			//serialize so there is an order id
			$order->serialize();
			$orderId = $order->getId();
			
			//insert into session table, so that we have a session ID
			$sql = "INSERT INTO ".geoTables::cart." (`session`, `user_id`, `admin_id`, `order`, `main_type`, `order_item`, `last_time`, `step`) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
			$query_data = array($sessionId, $userId, $adminId, $orderId,'cart',0, geoUtil::time(),'');
			$result = $this->db->Execute($sql, $query_data);
			
			if (!$result) {
				trigger_error('ERROR ORDER SQL: data: <pre>'.print_r($query_data, 1).'</pre> Sql: '.$sql.' Error Msg: '.$this->db->ErrorMsg());
				trigger_error('DEBUG CART: Return False');
				return false;
			}
			
			//make sure session vars are set..
			$this->cart_variables['id'] = $this->db->Insert_ID();
			$this->cart_variables['session'] = $sessionId;
			$this->cart_variables['user_id'] = $userId;
			$this->cart_variables['admin_id'] = $adminId;
			$this->cart_variables['order'] = $orderId;
			$this->cart_variables['main_type'] = $this->main_type;
			$this->cart_variables['order_item'] = 0;
			$this->cart_variables['last_time'] = geoUtil::time();
			$this->cart_variables['step'] = ''; //step not known yet, it's calculated later.
			
			//NOTE: current step is not known here, and that is on purpose...
			
			$vars = null; //set vars here if needed.
			geoOrderItem::callUpdate('geoCart_initSession_new',$vars);
		} else {
			//just restoring, but nothing to restore...
			$this->order = $this->item = false;
		}
		if (!$restoreOnly && $this->order) {
			$billing_info = $this->order->getBillingInfo();
			if ($billing_info && is_array($billing_info)) {
				//set billing info
				$this->user_data['billing_info'] = $billing_info;
			}
		}
		
		//it gets this far, the cart session is started up.
		return true;
	}
	
	/**
	 * Performs the given action, usually this is done in init()
	 * 
	 * @param string $action
	 */
	public function performAction($action)
	{
		trigger_error('DEBUG CART: Top of performAction('.$action.')');
		switch ($action) {
			case 'process':
				$this->action = 'process'; //process a cart page
				$this->_doProcess = true;
				break;
				
			case 'cancel':
				if (($this->main_type != 'cart' || $this->isStandaloneCart()) && is_object($this->item)) {
					//need to cancel adding the current item, so remove it from the cart
					//by running deleteProcess() which will remove the current item
					$this->action = 'cancel';
					$this->actions_performed[]['cancel'] = $this->item->getType();
					$this->deleteProcess();
				}
				
				break;
				
			case 'new':
				$main_type = false;
				
				if (isset($_GET['main_type']) && strlen($_GET['main_type']) > 0){
					$main_type = trim($_GET['main_type']);
				} else {
					//see if there is only one "button" that shows up, if there is,
					//use that button
					$buttons = geoOrderItem::callDisplay('geoCart_cartDisplay_newButton', null, 'string_array');
					if (count($buttons) == 1) {
						$types = array_keys($buttons);
						$main_type = $types[0];
					}
				}
				if ($main_type) {
					//see if it needs it's own cart
					
					$this->main_type = trim($main_type);
					
					//figure out login vars in case we need to pass them to login form
					require_once CLASSES_DIR . 'authenticate_class.php';
					$encodedUri = Auth::generateEncodedVars();
					
					if ($this->initItem(0, true, $encodedUri)){
						//initialized new item
						$this->cart_variables['main_type'] = $this->main_type;
						$this->action = 'new';
						$this->actions_performed[]['new'] = $this->main_type;
					} else {
						trigger_error('DEBUG CART: action new: init failed.');
						//init failed, set main type to cart
						$this->main_type = $this->cart_variables['main_type'] = 'cart';
						if ($this->cart_variables['order_item'] > 0){
							//do not set if 0 or -1 which is special case for items that require to be the only one in the checkout.
							$this->cart_variables['order_item'] = 0;
						}
						if ($this->cart_variables['order_item'] == 0){
							//only if the order item is not -1
							$this->item = null;
						}
					}
				}
				break;
				
			case 'edit':
				if (isset($_GET['item']) && is_numeric($_GET['item'])){
					$item_id = intval($_GET['item']);
					if ($item_id && $this->initItem($item_id,false)){
						$this->action = 'edit';
						$this->main_type = $this->cart_variables['main_type'] = $this->actions_performed[]['edit'] = $this->item->getType();
					}
				}
				break;
			
			case 'delete':
				if (isset($_GET['item']) && is_numeric($_GET['item'])){
					$item_id = intval($_GET['item']);
					$old_action = $this->action;
					$this->action = 'delete';
					$this->actions_performed[]['delete'] = $this->main_type;
					if ($item_id && $this->initItem($item_id)){
						$this->main_type = $this->cart_variables['main_type'] = $this->item->getType();
						$this->all_steps = array('delete', 'cart');
						$this->_skipInitSteps = true;
						$this->current_step = $this->cart_variables['step'] = 'delete';
						$this->_doProcess = true;//process deleting
					} else {
						$this->action = $old_action;
					}
				}
				break;
				
			case 'preview':
				if (isset($_GET['item']) && is_numeric($_GET['item'])){
					$item_id = intval($_GET['item']);
					//preserve current item
					
					if ($item_id){
						if ($this->item) {
							$oldItemId = $this->item->getId();
						}
						if ($this->initItem($item_id,false)) {
							$this->action = 'preview';
							$this->main_type = $this->cart_variables['main_type'] = $this->actions_performed[]['preview'] = $this->item->getType();
							$this->all_steps = array('preview','cart');
							$this->_skipInitSteps = true;
							$this->current_step = $this->cart_variables['step'] = 'preview';
							$this->_doProcess = false;
						}
					}
				}
				break;
				
			default:
				
				break;
		}
		trigger_error('DEBUG CART: End of performAction, action: '.$this->action);
		return true;
	}
	
	/**
	 * Initializes an item, either creating a new one or restoring an existing one
	 * in the cart, and sets {@link geoCart::item} to the item.
	 * 
	 * @param int $item_id The item id to restore, or if 0, will create a new item
	 *  with the type specified by the currently set {@link geoCart::main_type}  
	 * @param bool $force_parent if true, will make sure the item is a parent.
	 * @param bool|string $enforceAnon Only used if creating a new item.
	 *  If true (strict), will call {@link geoOrderItem::enforceAnonymous} passing
	 *   "a*is*cart" as first param.
	 *  If string, will call {@link geoOrderItem::enforceAnonymous} passing $enforceAnon
	 *   as first param.  Note that the empty string or null is allowed.
	 *  If false (strict), will not perform any anonymous checks.
	 * @return bool true if item was initialized, false otherwise.
	 */
	public function initItem ($item_id = 0, $force_parent = true, $enforceAnon = false)
	{
		//clean vars
		$item_id = intval($item_id);
		//create default order item
		$this->item = null;
		if ($item_id > 0) {
			//get item based on existing item id
			$existing_items = $this->order->getItem();
			foreach ($existing_items as $item) {
				if (is_object($item) && $item->getId() == $item_id) {
					//matches item in current order
					$this->item = $item;
					if (method_exists($item,'geoCart_initItem_restore')) {
						if (!$this->item->geoCart_initItem_restore()) {
							//the function called above must have decided to end it's own life early, so
							//don't proceed with initializing it.
							$this->item = null;
							$this->main_type = $this->cart_variables['main_type'] = 'cart';
							$this->cart_variables['order_item'] = 0;
							return false;
						}
					}
					if ($this->item->geoCart_initItem_forceOutsideCart()) {
						$this->cart_variables['order_item'] = -1;
					} else {
						$this->cart_variables['order_item'] = $this->item->getId();
					}
					//if price plan and category are set, set the cart's price plan and category
					$price_plan = intval($this->item->getPricePlan());
					if ($price_plan) {
						$category = intval($this->item->getCategory());
						$this->setPricePlan($price_plan, $category);
					}
					return true;
				}
			}
			trigger_error('DEBUG CART: Init Item Return False, item id: '.$item_id.' order items: <pre>'.print_r($existing_items,1).'</pre>');
			return false;
		}
		//id not set, so must be new item
		if ($this->cart_variables['order_item'] == -1) {
			trigger_error('DEBUG CART: Returning false in initItem, cannot create new item if cart is set to order_item = -1.');
			return false;
		}
		if (!$this->main_type || $this->main_type == 'cart') {
			//main type not set, can't create a new of nothing!
			trigger_error('DEBUG CART: Returning false in initItem, cannot create new item if main type not set or set to cart.');
			return false;
		}
		//enforce whether needs to be logged in
		if ($enforceAnon !== false) {
			//NOTE: DO NOT change surrounding if statement to "if ($enforceAnon)", as 
			//this will not allow an empty string to be used to pass to the enforce
			//anonymous function!
			
			//if enforceAnon is not a strict true, use that to pass to enforceAnonymous call,
			//otherwise use a*is*cart
			$loginVar = (($enforceAnon===true)? 'a*is*cart' : ''.$enforceAnon);
			
			if (geoOrderItem::enforceAnonymous($this->main_type, $loginVar)) {
				//no-anonymous allow was just enforced, login page was displayed,
				//so we need to exit.  Do not pass go.  Do not collect 200.
				include GEO_BASE_DIR . 'app_bottom.php';
				exit;
			}
		}
		
		$item = geoOrderItem::getOrderItem($this->main_type);
		if (!(is_object($item) && $item->getType() == $this->main_type && ($force_parent && count(geoOrderItem::getParentTypesFor($this->main_type)) == 0))) {
			trigger_error('DEBUG CART: Init Item Return False, main type: '.$this->main_type.' item: <pre>'.print_r($item,1).'</pre>');
			return false;
		}
		
		$this->item = $item;
		if (method_exists($this->item,'geoCart_initItem_new')) {
			if (!$this->item->geoCart_initItem_new()) {
				//the function called above must have decided to end it's own life early, so
				//don't proceed with initializing it.
				$id = $this->item->getId();
				if ($id) {
					geoOrderItem::remove($id);
					//detach from order, just in case it was added by something
					//internal to the order item
					$this->order->detachItem($id);
				}
				$this->item = null;
				$this->main_type = $this->cart_variables['main_type'] = 'cart';
				$this->cart_variables['order_item'] = 0;
				return false;
			}
		}
		//item is good, attach new blank item to order and set up all teh stuff for it.
		if ($this->item->geoCart_initItem_forceOutsideCart()) {
			//this is a new item, and the item specifies to be the only one...
			return $this->_initNewStandaloneCart($item);
		}
		//attach it to the order
		$item->setOrder($this->order);
		$this->order->addItem($item);
		$this->order->save();
		$this->cart_variables['order_item'] = $this->item->getId();
		return true;
	}
	/**
	 * Displays the message saying that they are already in the middle of doing something, and gives
	 * them the option to either continue with it, or to cancel and remove it, and start on the new
	 * thing they are trying to do.
	 * 
	 */
	public function displayCartInterruption ()
	{
		/*$msgs = $this->db->get_text(true, 10202);
		$this->addErrorMsg('cart_error',$msgs[500258]);
		*/
		
		$this->site->page_id = 10202;
		$this->site->get_text();
		
		$view = geoView::getInstance();
		$tpl_vars = $this->getCommonTemplateVars();
		
		$vars = array('action' => 'interrupted', 'step' => $this->cart_variables['step']);
		$action = geoOrderItem::callDisplay('getActionName',$vars,'',$this->main_type);
		
		$tpl_vars['interrupted_action'] = ($action)? $action: $cart->site->messages[500572];
		
		$vars = array ('action' => $_GET['action'], 'step' => $_GET['step']);
		if (isset($_GET['main_type']) && strlen(trim($_GET['main_type'])) > 0) {
			$action = geoOrderItem::callDisplay('getActionName',$vars,'',$_GET['main_type']);
		} else {
			$action = '';
		}
		//figure out the URL for the new action
		if (defined('IN_ADMIN')) {
			$url = 'index.php?page=admin_cart&amp;userId='.$this->user_data['id'].'&amp;action_special=cancel_and_go';
		} else {
			$url = $this->db->get_site_setting('classifieds_file_name').'?a=cart&amp;action_special=cancel_and_go';
		}
		foreach ($_GET as $key => $value) {
			$ignore = array('a','action_special');
			if (!in_array($key,$ignore)) {
				$url .= "&amp;{$key}=$value";
			}
		}
		
		
		$tpl_vars['new_action'] = ($action)? $action: $cart->site->messages[500571];
		$tpl_vars['new_action_url'] = $url;
		
		$view->setBodyTpl('display_cart/action_interrupted.tpl','','cart')
			->setBodyVar($tpl_vars);
		
		$this->site->display_page();
		return;
	}
	private function _initNewStandaloneCart($item)
	{
		//make sure the current order has nothing in it.
		trigger_error('DEBUG CART: Top of initNewStandaloneCart.');
		if ($this->cart_variables['order_item'] == -1) {
			trigger_error('DEBUG CART: Returning false, current order_item is -1 so can\'t add to this cart..');
			return false;
		} 
		
		if (!is_object($this->order)) {
			trigger_error('DEBUG CART: Returning false, order is not an object.');
			return false;
		}
		
		//need to create new cart!  Duplicate the initSession but with a few changes specific for stand-alone carts
		$this->order = null;
		trigger_error('DEBUG CART: Creating new stand-along cart session.');
		
		//create new order for this cart
		//create order object
		$order = new geoOrder();
		
		//set up order's info, if the individual order item wants to change any of these, they can.
		$order->setSeller(0);//seller is 0, it is the site doing the "selling", selling the ability to place the listing.
		$order->setBuyer($this->user_data['id']); //set buyer to be this user
		$order->setParent(0);//this is main listing order
		$order->setCreated(geoUtil::time());
		
		//serialize so there is an order id
		$order->serialize();
		$order_id = $order->getId();
		
		$this->order = $order;
		
		$this->item = $item;
		$this->item->setOrder($this->order);
		$this->order->addItem($this->item);
		//set main type
		$this->main_type = $this->item->getType();//'cart';
		$userId = $this->user_data['id'];
		$adminId = (defined('IN_ADMIN'))? $this->session->getUserId() : 0;
		$sessionId = ($userId > 0 || defined('IN_ADMIN'))? 0: $this->session->getSessionId();
		
		//insert into session table, so that we have a session ID
		$sql = "INSERT INTO ".geoTables::cart." (`session`, `user_id`, `admin_id`, `order`, `main_type`, `order_item`, `last_time`, `step`) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
		//set order_item to -1 to indicate this is stand-alone.
		$query_data = array($sessionId, $userId, $adminId, $this->order->getId(),'cart',-1, geoUtil::time(),'');
		$result = $this->db->Execute($sql, $query_data);
		
		if (!$result) {
			trigger_error('ERROR ORDER SQL: data: <pre>'.print_r($query_data, 1).'</pre> Sql: '.$sql.' Error Msg: '.$this->db->ErrorMsg());
			trigger_error('DEBUG CART: Return False');
			return false;
		}
		
		//make sure session vars are set..
		$this->cart_variables['id'] = $this->db->Insert_ID();
		$this->cart_variables['session'] = $sessionId;
		$this->cart_variables['user_id'] = $userId;
		$this->cart_variables['order'] = $order_id;
		$this->cart_variables['main_type'] = $this->main_type;
		$this->cart_variables['last_time'] = geoUtil::time();
		$this->cart_variables['step'] = '';
		$this->cart_variables['order_item'] = -1;
		return true;
	}
	
	/**
	 * Best way to tell if the cart is currently in "standalone" mode, meaning
	 * there can only be 1 thing in the cart right now, and that thing is already
	 * in there.
	 * 
	 * @return bool
	 */
	public function isStandaloneCart ()
	{
		return ($this->cart_variables['order_item'] == -1);
	}
	
	/**
	 * Returns whether or not the cart is specifically a recurring cart or not.
	 * 
	 * @return bool
	 * @since Version 4.1.0
	 */
	public function isRecurringCart ()
	{
		return ($this->isRecurringPossible() && $this->isStandaloneCart() && $this->item && $this->item->isRecurring());
	}
	
	/**
	 * Checks to see if recurring billing is even possible given the current
	 * enabled payment gateways and whether any of them can handle processing
	 * recurring billing.
	 * 
	 * @return bool
	 * @since Version 4.1.0
	 */
	public function isRecurringPossible ()
	{
		//checks to see if there are any payment gateways that are recurring
		$gateways = geoPaymentGateway::getPaymentGatewayOfType('recurring');
		return ($gateways && count($gateways) > 0);
	}
	
	protected function initSteps(){
		//make sure step is good
		$session_step = $this->cart_variables['step'];
		
		//initialize all steps if not already started.
		$this->all_steps = (isset($this->all_steps) && is_array($this->all_steps))? $this->all_steps: array();
		if (($this->action == 'edit' || $this->action == 'new' || $session_step !== 'cart') && $this->main_type !== 'cart' && is_object($this->item)) {
			//let the main order set the steps, it must get an instance of geoCart and set $cart->all_steps = array ()
			trigger_error('DEBUG CART: Steps being set by item.');
			
			geoOrderItem::callUpdate('geoCart_initSteps',null,$this->item->getType());
			
			$this->all_steps = $this->_validateStep($this->all_steps); //make sure all steps set are valid
			if (!is_array($this->all_steps)) {
				//Doh!  didn't return expected result...
				if (strlen($this->all_steps) && $this->_validateStep($this->all_steps)) {
					$this->all_steps = array ($this->all_steps); //all_steps was set to a string, but string is valid step so just convert it to array
				} else {
					$this->all_steps =array(); //just in case, init all steps to have no non-built in steps
				}
			}
			if (geoOrderItem::callDisplay('geoCart_initSteps_addOtherDetails',null,'bool_true',$this->main_type)) {
				$this->all_steps[] = 'other_details';	//collect misc. settings, built-in step
			}
			//last step of process, the cart view.
			$this->all_steps[] = 'cart'; //display contents of cart
		} else {
			trigger_error('DEBUG CART: Not doing anything with item, using built in cart steps.');
			
			$this->main_type = $this->cart_variables['main_type'] = 'cart';
			$this->all_steps = array();
			//main type is cart, make sure item is not still set
			if ($this->cart_variables['order_item'] != -1) {
				$this->item = null;
			}
			$this->cart_variables['order_item'] = (($this->cart_variables['order_item'] >= 0)? 0: -1);
			//for these steps, main_type = cart
			//add built-in steps - in cart, so first step is cart view
			$this->all_steps[] = 'cart'; //display contents of cart
			if ($this->getCartTotal() > 0 || $this->get('no_free_cart')) {
				//only if there is something to pay for
				$this->all_steps[] = 'payment_choices'; //enter details of payment
			}
			if (is_array($this->order->getItem()) && count($this->order->getItem())) {
				$this->all_steps[] = 'process_order'; //sends payment through payment gateway to process, only if there are items in the cart.
			}
		}
		if ($this->action == 'edit' || $this->action == 'new') {
			//if editing or adding new item, it has to be the first step.
			$this->current_step = $this->cart_variables['step'] = $this->all_steps[0];
			$this->_doProcess = false;
			$this->initStepsView();
			return true;
		}
		if (!in_array($session_step,$this->all_steps)) {
			//just in case session gets weird
			$session_step = $this->all_steps[0];
		}
		//step = Calculate current step.
		
		//Allow for linking to previous steps to edit things
		
		$force = (isset($_GET['step']))? $_GET['step']: 0;
		$step = 0;
		$force_used = 0;
		if (!$force) {
			//no need to go through all that if there is nothing set in the URL
			$step = $session_step;
		} else {
			//go through each of the allowed steps, and when we get to either the one set in the
			//url, or the one set in the session, make that one be the one used.
			//Since it goes in order, it will prevent people trying to go forward and skipping steps,
			//they will only be able to go backwards from what is set for their session.
			foreach ($this->all_steps as $s){
				if ($session_step == $s){
					$step = ($force_used)? $step: $s;
					break; //step from session found, don't go further
				}
				if ($force_used){
					//count how many far back
					$forced_used ++;
				} elseif ($force == $s){
					//the forced step needs to be before the current step set in the session
					//and it is, if it got to this
					$step = $s;
					$force_used = 1;
					//Keep going in loop, to see how far back this is from the main one.
				}
			}
		}
		if (!$step) {
			//Err step not found?  This should be impossible to get to, there has to be some logic problem...
			$this->initStepsView(); //still let view know current steps...
			return false;
		}
		$this->current_step = $this->cart_variables['step'] = $step;
		//figure out whether or not to check vars and process, or to just display..
		if ($this->action == 'process' && (!$force || $force == $session_step || ($force_used && $force_used < 2) )) {
			//only checkVars/process if: 
			//action == process AND
			//(
			//	step not set in URL OR
			//	step in URL is same as in session OR
			//	(
			//		step in URL is step being used AND
			//		step in URL is less than 2 before the step set in session
			//	)
			//)
			$this->_doProcess = true;
		}
		$this->initStepsView();
		return true;
	}
	private $_stepLabels = array();
	/**
	 * Sets the steps info in the geoView class, so that templates can display
	 * current step and progress.
	 */
	protected function initStepsView ()
	{
		//Loop through all the steps
		$viewSteps = array();
		//get text for main cart, most items have text for steps set in this page
		$this->site->messages = $this->db->get_text(true,10202);
		foreach ($this->all_steps as $key => $step) {
			$label = false;
			if (isset($this->_stepLabels[$step])) {
				$label = $this->_stepLabels[$step];
			} else {
				if (in_array($step, $this->built_in_steps)) {
					$call = $step.'Label';
					$label = $this->$call();
				} else {
					$parts = $this->_getStepParts($step);
					$name = $parts['item_name'];
					$realStep = $parts['step'];
					
					$label = geoOrderItem::callDisplay($realStep.'Label',null,'not_null',$name);
					if ($label === null) {
						//was not defined, or did not return anything, so reset label
						//to false, so that it will be auto-generated as a fallback
						$label = false;
					}
				}
				if ($label === false) {
					$parts = $this->_getStepParts($step);
					$name = $parts['item_name'];
					$realStep = $parts['step'];
					$label = ucwords(str_replace('_',' ',$realStep));
				}
			}
			
			$viewSteps[$step] = $label;
			$this->_stepLabels[$step] = $label;
		}
		$view = geoView::getInstance();
		$view->cartSteps = $viewSteps;
		$view->currentCartStep = $this->current_step;
	}
	
	/**
	 * Add a step to the cart system, with the option to insert the step
	 * before or after an already added step, or at the beginning of all
	 * the currently added steps.
	 * 
	 * Designed to be used in order items, in the init steps function.
	 *
	 * @param string $step The step to add, in the format item_name:step_name
	 * @param string $where
	 * @param string $otherStep
	 */
	public function addStep ($step, $where = self::LAST_STEP, $otherStep = null)
	{
		if ($this->_validateStep($step)) {
			switch ($where) {
				case self::FIRST_STEP:
					//push step to the from of the steps
					array_unshift($this->all_steps, $step);
					break;
					
				case self::AFTER_STEP:
					//insert step after specified $otherStep, if $otherStep is
					//already added.
					if (in_array($otherStep, $this->all_steps)) {
						$all = array();
						foreach ($this->all_steps as $key => $val) {
							$all [] = $val;
							if ($val == $otherStep) {
								$all[] = $step;
							}
						}
						$this->all_steps = $all;
					}
					break;
					
				case self::BEFORE_STEP:
					//insert step before specified $otherStep, if $otherStep is
					//already added.
					if (in_array($otherStep, $this->all_steps)) {
						$all = array();
						foreach ($this->all_steps as $key => $val) {
							if ($val == $otherStep) {
								$all[] = $step;
							}
							$all [] = $val;
						}
						$this->all_steps = $all;
					}
					break;
					
				case self::REPLACE_STEP:
					//replaces an existing step
					if (in_array($otherStep, $this->all_steps)) {
						$all = array();
						foreach ($this->all_steps as $key => $val) {
							$all[] = ($val == $otherStep)? $step : $val;
						}
						$this->all_steps = $all;
					}
					break;
				
				case self::LAST_STEP:
					//break ommited on purpose
					
				default:
					//Normal case, just add step to the end of the list
					$this->all_steps[] = $step;
					break;
			}
		}
	}
	
	/**
	 * Gets the next step after the current one.
	 * @return string
	 */
	public function getNextStep()
	{
		$run_next = 0;
		foreach ($this->all_steps as $s){
			if ($run_next){
				//this is the next step in the list.
				return ($s);
			}
			if ($s == $this->current_step){
				//this is current step, so the next one is the one to actually display
				$run_next = 1;
				trigger_error('DEBUG CART: next step is it..'.$s);
			}
		}
		//no next step, return current step?
		return $this->current_step;
	}
	
	/**
	 * NOT Meant for use in initSteps functions in order items, use
	 * addStep for that.
	 * 
	 * Inserts a step right before the current one, then sets it up
	 * so that the inserted step will be the next called step when
	 * geoCart::getNextStep() is called.
	 * 
	 * Designed to be used in order items in the check vars or process steps
	 * in order to change what step is actually displayed.
	 * 
	 * @param string $step
	 */
	public function insertStep($step)
	{
		$steps = $this->all_steps;
		$current = $this->current_step;

		//find index of current step
		$location = array_keys($steps, $current);
		$current_key = $location[0];
		
		//save everything after it
		$end = array_splice($steps, $current_key);
		

		//if array now empty, pad it so we don't break things
		if(!count($steps)) {
			$steps[] = "cart";
		}
		
		//append new step
		$steps[] = $step;
		
		//append saved stuff
		$steps = array_merge($steps, $end);
		
		//make changes to class vars
		$this->all_steps = $steps;
		
		//make the system think it just did the step before the one it actually did
		//so it getNextStep's into the one we just added
		$current_key--;
		if($current_key < 0) {
			$current_key = 0;
		}
		$this->current_step = $steps[$current_key];
		
		$this->cart_variables['step'] = $this->current_step;
		$this->save();			
	}
	
	/**
	 * Used during checkVars to signify there is a problem, so do not proceed to next step yet.
	 * 
	 * Can also be used in process section, but discouragede except in situations like when charging
	 * a credit card, and the transaction doesn't go through
	 *
	 * @return geoCart For easy chaining.
	 */
	public function addError()
	{
		//throw new Exception('yo'); //see where is adding an error
		$this->errors++;
		return $this;
	}
	
	/**
	 * Add an error message that can be used in other areas.  Typically used in checkVars step to record
	 * what the error was, so that in display step it can display an appropriate message.
	 * 
	 * Note that using this function alone DOES NOT prevent it from proceeding to the next step, you will
	 * need to use addError() to keep it from proceeding to the next step, and addErrorMsg to let the display
	 * step know what is wrong.
	 *
	 * @param string $error_name How error message is accessed, handy to specify an error message to be displayed
	 *  next to a specific input field
	 * @param string $msg
	 */
	public function addErrorMsg($error_name,$msg)
	{
		$this->error_msgs[$error_name] = $msg;
	}
	
	/**
	 * Gets the specified item from the Cart's registry.  Should only be used for settings that
	 * are global to the entire cart.  This is not typical, usually you would set the setting
	 * on an individual order item, not the entire cart.
	 *
	 * @param string $item
	 * @return Mixed the specified item, or false if item is not found.
	 */
	public function get($setting, $default = false)
	{
		$this->_initReg();
		return $this->registry->get($setting, $default);
	}
	
	/**
	 * Sets the given item to the given value.  If item is one of built-in items, it sets that instead
	 *  of something from the registry.
	 *
	 * @param string $item
	 * @param mixed $value
	 */
	public function set($item, $value)
	{
		$this->_initReg();
		return $this->registry->set($item, $value);
	}
	
	/**
	 * Checks the input vars for problems, like if user didn't fill in a required field.
	 * 
	 * Acutally it leaves it up to each order item to do it.
	 * 
	 * @param string $step
	 * @return bool true if everything is good, false otherwise.
	 */
	public function checkVars ($step = '')
	{
		if (strlen($step) > 0 && $this->_validateStep($step)) {
			//set current step to the step
			$this->current_step = $step;
		}
		
		if (in_array($this->current_step,$this->built_in_steps)) {
			//built in step
			$function_name = $this->current_step . 'CheckVars';
			$this->$function_name();
		} else {
			//Check vars specific to the current step
			$parts = $this->_getStepParts($this->current_step);
			$function_name = $parts['step'].'CheckVars';
			$item_name = $parts['item_name'];
			geoOrderItem::callUpdate($function_name,null,$item_name);
		}
		if (isset($this->site->error) && $this->site->error > 0) {
			$this->addError();
		}
		if ($this->errors > 0) {
			//errors generated, return false.
			return false;
		}
		//no errors generated, return true.
		return true;
	}
	
	/**
	 * Processes a step, or rather, calls the process step for the current order
	 * item.
	 * 
	 * @param string $step The step to process (will override the current step)
	 */
	public function processStep($step = '')
	{
		trigger_error('DEBUG CART: processStep');
		if (strlen($step) > 0 && $this->_validateStep($step)) {
			//set current step to the step
			$this->current_step = $step;
		}
		if (in_array($this->current_step,$this->built_in_steps)) {
			$function_name = $this->current_step . 'Process';
			$this->$function_name();
		} else {
			//Check vars specific to the current step
			$parts = $this->_getStepParts($this->current_step);
			$function_name = $parts['step'].'Process';
			$item_name = $parts['item_name'];
			
			geoOrderItem::callUpdate($function_name,null,$item_name);
		}
		
		//After processing, set step to be next step, as long as there is no errors
		//NOTE:  it is BAD PRACTICE to do var checking in process step, it only
		// checks for errors for things that could only go wrong when processing, for
		// instance if an error happens when processing a CC order.
		if ($this->errors == 0) {
			$this->current_step = $this->getNextStep();
			if ($this->current_step == 'cart') {
				//make sure to reset the steps, mostly for the benifit of the
				//display of the current steps in the cart
				$this->all_steps = array();
				//add built-in steps - in cart, so first step is cart view
				$this->all_steps[] = 'cart'; //display contents of cart
				if ($this->getCartTotal() > 0) {
					//only if there is something to pay for, add payment choices automatically
					$this->all_steps[] = 'payment_choices'; //enter details of payment
				}
				if (is_array($this->order->getItem()) && count($this->order->getItem())) {
					$this->all_steps[] = 'process_order'; //sends payment through payment gateway to process, only if there are items in the cart.
				}
			}
			if ($this->current_step != 'other_details' && in_array($this->current_step,$this->built_in_steps)) {
				//this is a built in step with no main item needed
				$this->main_type = 'cart';
				$this->cart_variables['main_type'] = 'cart';
				$this->cart_variables['order_item'] = (($this->cart_variables['order_item'] >= 0)? 0: -1);
			}
			$this->cart_variables['step'] = $this->current_step;
		}
	}
	
	/**
	 * Displays the current step, if a step is for an order item, hands it
	 * over to that order item to display the step.
	 */
	public function displayStep()
	{
		if ($this->current_step != 'preview') $this->initStepsView();
		if (in_array($this->current_step,$this->built_in_steps)) {
			//call the built in steps locally
			$function_name = $this->current_step . 'Display';
			
			$this->$function_name();
			
			//echo '<h1>Build in step: '.$this->current_step.'</h1><a href="'.$this->getProcessFormUrl().'">Next ></a><pre>'.print_r($this->order,1);
			return;
		}
		
		$parts = $this->_getStepParts($this->current_step);
		
		$function_name = $parts['step'].'Display';
		
		geoOrderItem::callUpdate($function_name, null, $parts['item_name']);
	}
	
	/**
	 * Special built-in step of "other details" aka listing extras, or any other
	 * misc. data needing to be collected, this checks the vars by calling
	 * each order item and letting it check the vars specific to it.
	 * 
	 */
	public function other_detailsCheckVars()
	{
		trigger_error('DEBUG CART: Running other_detailsCheckVars');
//		$this->site->page_id = 12;
//		$this->site->get_text();
		
		$specific_item = null;
		if (count(geoOrderItem::getParentTypesFor($this->item->getType())) > 0) {
			//this is a child which won't be auto called by parent,
			//so force it to call
			$specific_item = $this->item->getType();
		}
		geoOrderItem::callUpdate('geoCart_other_detailsCheckVars', null, $specific_item);
		//$this->errors ++;//force there to be errors
	}
	
	/**
	 * Special built-in step of "other details" aka listing extras, or any other
	 * misc. data needing to be collected, this processes the vars by calling
	 * each order item and letting it process the vars specific to it.
	 * 
	 */
	public function other_detailsProcess()
	{
		trigger_error('DEBUG CART: Running other_detailsProcess');
		
		$specific_item = null;
		if (count(geoOrderItem::getParentTypesFor($this->item->getType())) > 0) {
			//this is a child which won't be auto called by parent,
			//so force it to call
			$specific_item = $this->item->getType();
		}
		geoOrderItem::callUpdate('geoCart_other_detailsProcess', null, $specific_item);
	}
	
	/**
	 * Special built-in step of "other details" aka listing extras, or any other
	 * misc. data needing to be collected, this displays the step by calling
	 * each order item and letting it send in stuff to be displayed on the page.
	 * 
	 * See the _template order item for further documentation.
	 * 
	 */
	public function other_detailsDisplay ($return = false)
	{
		//---------- LISTING COST AND FEATURES -------------
		
		/**
		 * Expects each one to return an associative array, like so:
		 * array (
		 *  'checkbox_name' => 'string', //if empty string, no checkbox is displayed
		 * 	'title' => 'string',
		 * 	'display_help_link' => 'string', //the help link returned by display_help_link($link_id);
		 * 	'price_display' => 'string', //IGNORED if charge for listings is turned off.
		 * 	//templates - over-write mini-template to do things like set margine or something:
		 * 	'entire_box' => 'inside of box actually',
		 * 	'left' => 'left side part',
		 * 	'right' => 'right side part',
		 * 	'checkbox' => 'checkbox tag',
		 * 	'checkbox_hidden' => 'hidden input tag',
		 *  //THE FOLLOWING are only needed by itens that will potentially be
		 *  // the "main_type" when this page is loading.  Such as any items
		 *  // that are parents, or any items that use this page to set settings
		 *  // and have ability to edit (like attention getters)
		 *  'page_title1' => 'string', //title with lines on it
		 *  'page_title2' => 'string', //blue title
		 *  'page_desc' => 'string',
		 *  'submit_button_text' => 'string',
		 *  'cancel_text' => 'string', 
		 * )
		 */
		if ($return) {
			$this->site->messages = $msgs = $this->db->get_text(true, 10205);
		} else {
			$this->site->page_id = 10205;
			$this->site->get_text();
			$msgs = $this->db->get_text(true, $this->site->page_id);
		}
		
		$data_raw = geoOrderItem::callDisplay('geoCart_other_detailsDisplay',null,'array','',true);
		if (count($data_raw) > 0) {
			$tpl_vars = $this->getCommonTemplateVars();
			$mainData = (isset($data_raw[$this->main_type]))? $data_raw[$this->main_type]: array();
			//$tpl = new geoTemplate('system','cart');
			$tpl_vars['allFree'] = $this->db->get_site_setting('all_ads_are_free');
			$tpl_vars['page_title1'] = (isset($mainData['page_title1']))? $mainData['page_title1']: $msgs[500311];
			$tpl_vars['page_title2'] = (isset($mainData['page_title2']))? $mainData['page_title2']: $msgs[500311];
			
			$tpl_vars['page_desc'] = (isset($mainData['page_desc']))? $mainData['page_desc']: $msgs[500312];
			
			$tpl_vars['submit_button_text'] = (isset($mainData['submit_button_text']))? $mainData['submit_button_text']: $msgs[500397];
			$tpl_vars['cancel_text'] = (isset($mainData['cancel_text']))? $mainData['cancel_text']: $msgs[500310];
			
			$tpl_vars['form_url'] = $this->getProcessFormUrl();
			$tpl_vars['cancel_url'] = $tpl_vars['cart_url'].'&amp;action=cancel';
			
			$tpl_vars['items'] = $data_raw;
			$tpl_vars['error_msgs'] = $this->getErrorMsgs();
			
			if ($return) {
				$tpl = new geoTemplate('system','cart');
				$tpl->assign($tpl_vars);
				$tpl->full_step = 1;
				return $tpl->fetch('other_details/index.tpl');
			} else {
				geoView::getInstance()->setBodyTpl('other_details/index.tpl','','cart')
					->setBodyVar($tpl_vars);
			}
			
			$this->site->display_page();
		} else {
			//it just so happens nothing is supposed to display
			
			//if something wanted to still do the page and force it to be done,
			//it would need to return something, otherwise it assumes that
			//we really didn't want to do this step.
			
			if ($return) {
				return '';
			}
			
			$this->current_step = $this->cart_variables['step'] = $this->getNextStep();
			if ($this->current_step != 'other_details') {
				//re-display step
				$this->displayStep();
			} else {
				trigger_error("DEBUG CART: shouldn't be here -- something's wrong.");
			}
		}
		return true;
	}
	
	/**
	 * Mimics the other details step, but this is only used to display the info
	 * for specific item.
	 * 
	 * Designed so that you put your main settings to be set in other details
	 * step, then use this function if that info needs to be displayed on another
	 * step as well.
	 * 
	 * This function calls display_page if everything goes well.
	 * 
	 * @param string $item_type - item type to call on to get the data for displaying
	 *  on the other details page.
	 * @return unknown_type
	 */
	public function displaySingleOtherDetails ($item_type)
	{
		$this->site->page_id = 10205;
		$this->site->get_text();
		$msgs = $this->db->get_text(true, $this->site->page_id);
		$data = geoOrderItem::callDisplay('geoCart_other_detailsDisplay',null,'array',$item_type);
		if (count($data) > 0) {
			$tpl_vars = $this->getCommonTemplateVars();
			$mainData = (isset($data[$item_type]))? $data[$item_type]: array();
			//$tpl = new geoTemplate('system','cart');
			$tpl_vars['allFree'] = $this->db->get_site_setting('all_ads_are_free');
			$tpl_vars['page_title1'] = (isset($mainData['page_title1']))? $mainData['page_title1']: $msgs[500311];
			$tpl_vars['page_title2'] = (isset($mainData['page_title2']))? $mainData['page_title2']: $msgs[500311];
			
			$tpl_vars['page_desc'] = (isset($mainData['page_desc']))? $mainData['page_desc']: $msgs[500312];
			
			$tpl_vars['submit_button_text'] = (isset($mainData['submit_button_text']))? $mainData['submit_button_text']: $msgs[500397];
			$tpl_vars['cancel_text'] = (isset($mainData['cancel_text']))? $mainData['cancel_text']: $msgs[500310];
			
			$tpl_vars['form_url'] = $this->getProcessFormUrl();
			
			$tpl_vars['cancel_url'] = $tpl_vars['cart_url'].'&amp;action=cancel';
			
			$tpl_vars['items'] = $data;
			$tpl_vars['error_msgs'] = $this->getErrorMsgs();
			geoView::getInstance()->setBodyTpl('other_details/index.tpl','','cart')
				->setBodyVar($tpl_vars);
			
			$this->site->display_page();
			return true;
		} else {
			//it just so happens nothing is supposed to display
			//return false to let em know nothing happened.
			return false;
		}
	}
	
	/**
	 * Checks the vars (or lets the order items check any vars they want to) for
	 * the built-in main cart display page.
	 * 
	 */
	public function cartCheckVars()
	{
		//set text first for anything that might need text for error messages or something
		$this->site->page_id = 10202;
		$this->site->get_text();
		geoOrderItem::callUpdate('geoCart_cartCheckVars');
	}
	
	/**
	 * Processes the vars (or lets the order items do any processing they want to) for
	 * the built-in main cart display page.
	 * 
	 */
	public function cartProcess ()
	{
		geoOrderItem::callUpdate('geoCart_cartProcess');
		
		if ($this->errors > 0) {
			//don't "short circuit" payment page if something threw an error...
			return;
		}
		if ($this->getCartTotal() == 0 && !$this->get('no_free_cart')) {
			//there will be no payment details page, need to do things at this step.
			$this->set('free_cart',1);
			$this->payment_choicesProcess(true);
		} else {
			if ($this->get('free_cart')) {
				$this->set('free_cart', false);
			}
		}
	}
	/**
	 * Displays the main built-in cart display page.
	 * @param bool $renderPage If false, will skip actually displaying the page.
	 *   Useful if needing to set everything up for a cart to be displayed,
	 *   without actually displaying the cart.
	 */
	public function cartDisplay ($renderPage = true)
	{
		//Display the cart...
		
		if ($renderPage) {
			//make sure type and all that are reset
			$this->main_type = $this->cart_variables['main_type'] = 'cart';
			$this->cart_variables['order_item'] = (($this->cart_variables['order_item'] >= 0)? 0: -1);
		}
		
		//assign id of old choose listing type page, for now...
		$this->site->page_id = 10202;
		$this->site->get_text();
		$tpl_vars = $this->getCommonTemplateVars();
		$tpl_vars['allFree'] = $allFree = $this->db->get_site_setting('all_ads_are_free');
		//for each item that is main type, call to let that item specify what to display
		$cart_items = $this->_getCartItemDetails($allFree);
		
		if ($this->cart_variables['order_item'] != -1) {
			$new_item_buttons = geoOrderItem::callDisplay('geoCart_cartDisplay_newButton',null,'string_array');
		} else {
			$new_item_buttons = array();
		}
		if ($renderPage) {
			if (count($cart_items) == 0 && count($new_item_buttons) == 0) {
				//no items in cart, and no add buttons to display, so make sure user logged in by enforceAnonymous
				if (geoOrderItem::enforceAnonymous()) {
					if (defined('IN_ADMIN')) {
						//force re-direct to choose user
						header ("Location: index.php?page=admin_cart_select_user");
					}
					include GEO_BASE_DIR . 'app_bottom.php';
					exit;
				}
			}
			$tpl_vars['items'] = $cart_items;
			$tpl_vars['new_item_buttons'] = $new_item_buttons;
			$tpl_vars['error_msgs'] = $this->error_msgs;
			
			geoView::getInstance()->setBodyTpl('display_cart/index.tpl','','cart')
				->setBodyVar($tpl_vars);
			
			$this->site->display_page();
		}
	}
	/**
	 * Gets the label to use for the cart.
	 * @return string
	 */
	public function cartLabel ()
	{
		if ($this->db->get_site_setting('all_ads_are_free')) {
			//free, use "queue" text
			return $this->site->messages[500511];
		}
		return $this->site->messages[500510];
	}
	
	/**
	 * The delete label, used for the delete step (is just a blank string)
	 * @return string
	 */
	public function deleteLabel ()
	{
		return '';
	}
	/**
	 * gets the label for the preview step, is just a blank string.
	 * @return string 
	 */
	public function previewLabel ()
	{
		return '';
	}
	
	/**
	 * Gets the label for the built in step of other details (aka extras).
	 * @return string
	 */
	public function other_detailsLabel()
	{
		$label = 'Extras';
		if ($this->item) {
			$type = $this->item->getType();
			$label = geoOrderItem::callDisplay('geoCart_other_detailsLabel',null,'not_null',$type);
			if ($label !== null) {
				return $label;
			}
		}
		return $label;
	}
	
	/**
	 * Gets the label for the payment choices step.
	 * @return string
	 */
	public function payment_choicesLabel()
	{
		if ($this->db->get_site_setting('all_ads_are_free') || ($this->getCartTotal() == 0 && !$this->get('no_free_cart'))) {
			//don't display this step, it will be skipped!
			return '';
		}
		//Verify/Payment Details
		return $this->site->messages[500512];
	}
	
	/**
	 * Gets the label for the process order step.
	 * @return string
	 */
	public function process_orderLabel ()
	{
		if ($this->db->get_site_setting('all_ads_are_free')) {
			return $this->site->messages[500514];
		}
		//Verify/Payment Details
		return $this->site->messages[500513];
	}
	
	private function _getCartItemDetails($noTotal = false, $inCart = true)
	{
		//allow special case items such as sub-total to update themself if needed,
		//based on contents of the cart at this step.
		geoOrderItem::callUpdate('geoCart_getCartItemDetails',null,'',true);
		$items = $this->order->sortItems()
			->getItem();
		//for each item that is main type, call to let that item specify what to display
		$cart_items = array();
		foreach ($items as $i => $item) {
			if (is_object($item)) {
				if (!is_object($item->getParent())) {
					//this is parent
					$result = $item->getDisplayDetails($inCart);
					if ($result !== false) {
						$cart_items[$item->getId()] = $result;
					}
				}
			}
		}
		if (!$noTotal && count($cart_items)) {
			//add total
			$cart_items[] = array(
				'css_class' => 'total_cart_item', //css class	
				'title' => $this->site->messages[500403],
				'canEdit' => false, //show edit button for item?
				'canDelete' => false, //show delete button for item?
				'canPreview' => false, //show preview button for item?
				'priceDisplay' => geoString::displayPrice($this->getCartTotal()), //Price as it is displayed
				'cost' => 0, //amount this adds to the total, what getCost returns
				'total' => $this->getCartTotal(), //amount this AND all children adds to the total (will add to it as we parse the children)
				'children' => array()
			);
		}
		return $cart_items;
	}
	
	/**
	 * This is the checkvars method for the built-in delete step.
	 */
	public function deleteCheckVars()
	{
		trigger_error('DEBUG CART: Running deleteCheckVars');
		
		geoOrderItem::callUpdate('geoCart_deleteCheckVars', null, $this->item->getType());
	}
	
	/**
	 * This is the process method for the built-in delete step.
	 */
	public function deleteProcess()
	{
		trigger_error('DEBUG CART: Running deleteProcess, about to remove item '.$this->item->getId().' from this order.');
		
		//allow items to do special removal, if needed.
		geoOrderItem::callUpdate('geoCart_deleteProcess',null,$this->item->getType());
		
		//remove the item
		$id = $this->item->getId();
		geoOrderItem::remove($id);
		$this->order->detachItem($id);
		if ($this->cart_variables['order_item'] == -1) {
			//Delete the stand-alone cart
			$this->removeSession();
			//init main cart
			$this->initSession();
			return true;
		}
		$this->item = $this->cart_variables['order_item'] = 0;
	}
	
	/**
	 * This is the display method for the built-in delete step.
	 */
	public function deleteDisplay()
	{
		$this->cartDisplay();
	}
	
	/**
	 * This is the display method for the built-in preview step.
	 */
	public function previewDisplay()
	{
		$this->item->geoCart_previewDisplay();
		//set everything back so that next time cart is viewed, it's in cart mode
		$this->cart_variables['step'] = $this->current_step = $this->cart_variables['main_type'] = 'cart';
		if ($this->cart_variables['order_item'] != -1) {
			$this->item = null;
		}
		$this->cart_variables['order_item'] = (($this->cart_variables['order_item'] >= 0)? 0: -1);
	}
	
	public function payment_choicesCheckVars()
	{
		$this->site->page_id = 10203;
		$msgs = $this->site->messages = $this->db->get_text(true, $this->site->page_id);
		
		
		//make sure payment choice is made..
		if (!isset($_POST['c']['payment_type']) || !$_POST['c']["payment_type"] || !is_object(geoPaymentGateway::getPaymentGateway($_POST['c']["payment_type"]))) {
			//payment type not send, or set to something invalid.
			$this->addError();
			$this->error_variables["choices_box"] = geoString::fromDB($msgs[500308]);
		}
		
		if (isset($_POST['c']) && is_array($_POST['c'])) {
			//set billing info so it is used the next time the page
			//is loaded, if something has failed.  (so user does not
			//have to re-enter info every time)
			
			$this->order->setBillingInfo($_POST['c']);
			$this->user_data['billing_info'] = $_POST['c'];
			
			//make sure e-mail is valid...
			if (!isset($_POST['c']['email']) || !$_POST['c']['email'] || !geoString::isEmail($_POST['c']['email'])) {
				$this->addError()
					->addErrorMsg('billing_email',$msgs[500309]);
			}
			if (isset($_POST['c']['payment_type']) && $this->isRecurringCart()) {
				$gateway = geoPaymentGateway::getPaymentGateway($_POST['c']['payment_type']);
				if ($gateway && $gateway->isRecurring() && $gateway->getRecurringAgreement()) {
					//make sure the recurring agreement was agreed upon
					if (!isset($_POST['c']['user_agreement']) || $_POST['c']['user_agreement'] != $_POST['c']['payment_type']) {
						//user did not agree!
						$this->addError()
							->addErrorMsg('gateway_recurring_agreement', $msgs[500766]);
					}
				}
			}
		}
		geoPaymentGateway::callUpdate('geoCart_payment_choicesCheckVars');
	}
	
	/**
	 * This is the process method for the built in payment_choices step, it 
	 * is the step that the invoice is created, site fees added, etc.
	 * 
	 * @param bool $free_cart If true, treats it as if there is nothing owed
	 *  for the cart.
	 */
	public function payment_choicesProcess($free_cart = false)
	{
		if (isset($_POST['c']['payment_type']) && is_object(geoPaymentGateway::getPaymentGateway($_POST['c']['payment_type']))) {
			//set the payment type
			$this->order->set('payment_type',$_POST['c']['payment_type']);
		} elseif ($free_cart) {
			$this->order->set('payment_type','free');
		}
		//create an invoice, if there isn't one already
		$invoice = $this->order->getInvoice();
		if (!is_object($invoice)) {
			$invoice = new geoInvoice;
			$invoice->setOrder($this->order);
			$invoice->save(); //so it has an ID
			
			$this->order->setInvoice($invoice);
		}
		$invoice->setCreated(geoUtil::time());
		$invoice->setDue(geoUtil::time());
		$gateway = geoPaymentGateway::getPaymentGateway('site_fee');
		if (!is_object($gateway)) {
			trigger_error('ERROR CART: Unable to get gateway for site fee, not able to process.');
			return false;
		}
		//do the built-in transaction for the entire amount
		$trans = $invoice->getTransaction();
		$transaction = null;
		if (count($trans) > 0) {
			foreach ($trans as $tran) {
				if (is_object($tran) && $tran->getGateway()->getName() == 'site_fee') {
					$transaction = $tran;
					break;
				}
			}
		}
		if (!is_object($transaction)) {
			$transaction = new geoTransaction;
			$transaction->setGateway($gateway);
			$transaction->setInvoice($invoice);
			$transaction->save();
			$invoice->addTransaction($transaction);
		}
		
		$msgs = $this->db->get_text(true, 10202);
		
		//the cart total is a positive amount for how much the user owes us, so to convert to
		//a transaction amount it needs to be negative, kind of like taking away from a bank account
		$transaction->setAmount(-1 * $this->getCartTotal());
		$transaction->setGateway($gateway);
		$transaction->setDate(geoUtil::time());
		$transaction->setDescription($msgs[500259].$this->order->getId());
		$transaction->setInvoice($invoice);
		$transaction->setStatus(1);//turn on
		$transaction->setUser($this->user_data['id']);
		
		//save changes, should remove this once the app_bottom auto-save is done.
		
		$transaction->save();
		$invoice->save();
		//first let order items get ahold of it, even child order items...
		geoOrderItem::callUpdate('geoCart_payment_choicesProcess',null,'',true);
		
		
		if ($free_cart) {
			//it's free, so let all the order items know theres a new transaction.
			
			$this->order->processStatusChange('active');
		} else {
			//let gateways do their thing, but only if there is money involved
			geoPaymentGateway::callUpdate('geoCart_payment_choicesProcess',null,$this->order->get('payment_type'));
		}
	}
	
	/**
	 * Displays the payment choices page.
	 */
	public function payment_choicesDisplay()
	{
		//get the text for cart page, since displaying a mini-cart within the payment details page
		$this->site->page_id = 10202;
		$this->site->get_text();
		
		$this->site->page_id = 10203;
		$this->site->get_text();
		
		$tpl_vars = $this->getCommonTemplateVars();
		
		$tpl_vars['error_msgs'] = $this->error_msgs;
		$tpl_vars['items'] = $this->_getCartItemDetails(false, false);
		
		//$tpl_vars['user'] = $this->user_data;
		if (isset($this->error_variables)) {
			$tpl_vars['errors'] = $this->error_variables;
		} else {
			$tpl_vars['errors'] = array();
		}
		$payment_choices = geoPaymentGateway::callDisplay('geoCart_payment_choicesDisplay',null,'array');
		if ($this->get('no_free_cart') && $this->getCartTotal() == 0) {
			$tpl_vars['no_free_cart'] = 1;
			
			//go through each gateway and see which ones will still be displayed
			foreach ($payment_choices as $type => $vals) {
				if (!geoPaymentGateway::callDisplay('geoCart_payment_choicesDisplay_freeCart', null, 'bool_true', $type)) {
					//this payment gateway doesn't want to be displayed when it's a free cart
					unset($payment_choices[$type]);
				}
			}
		} else if ($this->isRecurringCart()) {
			//this is a recurring cart, see if any of the payment gateways is
			//recurring
			$recurring_choices = array ();
			foreach ($payment_choices as $type => $vals) {
				//isRecurring is not static, have to call it directly
				$gateway = geoPaymentGateway::getPaymentGateway($type);
				if ($gateway->isRecurring()) {
					$vals['user_agreement'] = $gateway->getRecurringAgreement();
					$recurring_choices[$type] = $vals;
				}
				unset($gateway);
			}
			if (count($recurring_choices)) {
				//there is at least 1 recurring gateway choice, so only show those
				//gateway(s) that are recurring
				$payment_choices = $recurring_choices;
			}
		}
		
		$force_checked = false;
		if (count($payment_choices) == 1) {
			$force_checked = true;
		} elseif (isset($_POST['c']['payment_type']) && isset($payment_choices[$_POST['c']['payment_type']])) {
			//set checked to true, for one that is selected.
			$payment_choices[$_POST['c']['payment_type']]['checked'] = true;
		} elseif ($this->order->get('payment_type') && isset($payment_choices[$this->order->get('payment_type')])) {
			//set as cart variable, probably selected in previous attempt
			$payment_choices[$this->order->get('payment_type')]['checked'] = true;
		} else {
			//go through each one, if it's the default then set it
			$foundChecked = false;
			foreach ($payment_choices as $type => $vals) {
				$gateway = geoPaymentGateway::getPaymentGateway($type);
				if (is_object($gateway) && $gateway->getDefault()) {
					//this is the default one, so make it checked
					$payment_choices[$type]['checked'] = true;
					$foundChecked = true;
					break;
				}
			}
			
			if (!$foundChecked) {
				//make sure something is checked!  This is for fallback purposes, in case admin has not selected a default.
				$force_checked = true;//just make them all checked...
			}
		}
		$tpl_vars['populate_billing_info'] = $populate_billing_info = $this->db->get_site_setting('populate_billing_info');
		//die('<pre>Region:'.print_r($this->user_data,1).'</pre>');
		$region = geoRegion::getInstance();
		$country = $state = '';
		if (isset($this->user_data['billing_info']['country'])) {
			$country = $this->user_data['billing_info']['country'];
		} else if ($populate_billing_info && isset($this->user_data['country'])) {
			$country = $this->user_data['country'];
		}
		if (isset($this->user_data['billing_info']['state'])) {
			$state = $this->user_data['billing_info']['state'];
		} else if ($populate_billing_info && isset($this->user_data['state'])) {
			$state = $this->user_data['state'];
		}
		$tpl_vars['countries'] = $region->getRegion($country, 'c[country]', true);
		$tpl_vars['states'] = $region->getSubRegion($state);
		
		$tpl_vars['cart'] = $this->user_data;
		$tpl_vars['payment_choices'] = $payment_choices;
		$tpl_vars['force_checked'] = $force_checked;
		
		geoView::getInstance()->setBodyTpl('payment_choices/index.tpl','','cart')
			->setBodyVar($tpl_vars);
		
		$this->site->display_page();
		return true;
	}
	
	/**
	 * The built in display method for process order step.
	 */
	public function process_orderDisplay()
	{
		//get the text for cart page, since displaying a mini-cart within the payment details page
		$cart->site->messages = $this->db->get_text(true,10202);
		
		geoView::getInstance()->cart_items = $this->_getCartItemDetails(false,false);
		
		//A way for manual type gateways to display a page, or used as success/failure page
		if ($this->get('free_cart')) {
			//the whole cart was free!
			$this->site->page_id = 10204;
			$msgs = $this->db->get_text(true, $this->site->page_id);
			$allFree = $this->db->get_site_setting('all_ads_are_free');
			$title = ($allFree)? $msgs[500417]: $msgs[500306];
			$desc = ($allFree)? $msgs[500418]: $msgs[500307];
			$tpl_vars = $this->getCommonTemplateVars();
			$tpl_vars['page_title'] = $title;
			$tpl_vars['page_desc'] = $desc;
			$tpl_vars['success_failure_message'] = '';
			$tpl_vars['my_account_url'] = $this->db->get_site_setting('classifieds_file_name').'?a=4';
			$tpl_vars['my_account_link'] = $msgs[500305];
			
			geoView::getInstance()->setBodyTpl('shared/transaction_approved.tpl','','payment_gateways')
				->setBodyVar($tpl_vars);
			
			$this->site->display_page();
			$this->removeSession();
			return;
		}
		
		geoPaymentGateway::callUpdate('geoCart_process_orderDisplay',null,$this->order->get('payment_type'));
	}
	
	/**
	 * Saves the cart, the order attached to the cart, and all the stuff in that
	 * order.  This is normally done by app_bottom but will need to be done
	 * manually if you are calling removeSession as otherwise, the final order
	 * changes won't be able to be saved.
	 * 
	 * @return geoCart Returns instance of geoCart to allow method chaining
	 */
	public function save ()
	{
		trigger_error('DEBUG CART: Saving cart TOP');
		if (!$this->cart_variables['id']) {
			//nothing to save...
			trigger_error('DEBUG CART: NOTHING to save, no ID!');
			return $this;
		}
		$sql = "UPDATE ".geoTables::cart." SET `session`=?, `user_id`=?, `admin_id`=?, `order`=?, `main_type`=?, `order_item`=?, `last_time`=?, `step`=? WHERE `id`=? LIMIT 1";
		$query_data =array(
			''.$this->cart_variables['session'],
			(int)$this->cart_variables['user_id'],
			(int)$this->cart_variables['admin_id'],
			(int)$this->cart_variables['order'],
			''.$this->cart_variables['main_type'],
			(int)$this->cart_variables['order_item'],
			(int)$this->cart_variables['last_time'],
			''.$this->cart_variables['step'],
			(int)$this->cart_variables['id'],
		);
		$result = $this->db->Execute($sql, $query_data);
		if (!$result){
			trigger_error('ERROR SQL CART: Sql: '.$sql.' Error Msg: '.$this->db->ErrorMsg());
		}
		$this->_initReg();
		$this->registry->save();
		
		if (is_object($this->item)){
			trigger_error('DEBUG CART: Saving item');
			$this->item->save();
		}
		if (is_object($this->order)){
			trigger_error('DEBUG CART: Saving order, kinda weird that we save order after item...');
			$this->order->save();
		}
		return $this;
	}
	
	/**
	 * Static function ideal for getting rid of a particular cart, without
	 * having to instantiate the cart class.
	 * 
	 * @param int $id
	 */
	public static function remove ($id)
	{
		$id = intval($id);//clean input
		if (!$id) {
			return false;
		}
		
		//remove registry items too
		geoRegistry::remove('cart',$id);
		
		$db = DataAccess::getInstance();
		
		$sql = "DELETE FROM ".geoTables::cart." WHERE `id` = ? LIMIT 1";
		$result = $db->Execute($sql, array($id));
	}
	
	/**
	 * Used to remove a cart session, this is used when a cart is done, paid for,
	 * and needs to be cleared out to allow for more junk to be added to.
	 * 
	 * @param int $id If set, will remove the cart for the specified ID instead
	 *  of the one for the current user.
	 * @param bool $saveOrder Added in Version 4.0.9: if true (default) and no ID passed in, it will
	 *  automatically save the order attached to the cart before removing the
	 *  cart.
	 */
	public function removeSession($id = 0, $saveOrder = true)
	{
		if (!$id) {
			$id = $this->cart_variables['id'];
			if ($saveOrder && is_object($this->order)) {
				//save the order before we remove the session
				$this->order->save();
			}
			//make it not save in app_bottom
			$this->cart_variables['id'] = 0;
		}
		$id = intval($id);//clean input
		
		self::remove($id);
		
		//allow order items to do things specific to order item...
		geoOrderItem::callUpdate('geoCart_removeSession');
		
		//allow from payment gateways too
		geoPaymentGateway::callUpdate('geoCart_removeSession');
	}
	
	/**
	 * Gets the child order item as specified by the type name.  Gets the one that is a child of the current main order item.
	 * Requires that order and order items are serialized so that they have an ID already.
	 * 
	 * Returns null if no child item by that name could be found.
	 *
	 * @param string $item_name
	 * @return geoOrderItem
	 */
	public function getChildItem($item_name)
	{
		return geoOrderItem::getOrderItemFromParent($this->item, $item_name);
	}
	
	/**
	 * Generates the URL to be used in a form tag, that would allow the form to submit so that the current page gets processed.
	 * Note that this is ONLY the URL, it is not the entire FORM tag. It automatically accounts for the SSL setting in the 
	 * admin turned on or off, as long as $ssl is left to be true.  If $ssl is set to false, it will use the non-ssl URL
	 * even if SSL is turned on in the admin.
	 *
	 * @param boolean $ssl
	 * @param boolean $onlyCart If true, will only return base URL plus a=cart.
	 * @return string
	 */
	public function getProcessFormUrl ($ssl = true, $onlyCart = false)
	{
		//preserve sub-domain
		$base = geoFilter::getBaseHref();
		
		$vars = array();
		
		if (defined('IN_ADMIN')) {
			$url = $base . ADMIN_LOCAL_DIR . 'index.php';
			$vars[] = 'page=admin_cart';
			$vars[] = 'userId='.(int)$this->user_data['id'];
		} else {
			//figure out if should be in SSL or not...
			$http = ($ssl && $this->db->get_site_setting('use_ssl_in_sell_process'))? 'https' : 'http';
			$url = preg_replace('/^https?/', $http, $base).$this->db->get_site_setting('classifieds_file_name');
			
			$vars[] = 'a=cart';
		}
		
		if (!$onlyCart) {
			$vars[] = 'action=process';
			if (isset($this->main_type)) {
				$vars[] = 'main_type='.urlencode($this->main_type);
			}
			if (isset($this->current_step)) {
				$vars[] = 'step='.urlencode($this->current_step);
			}
		}
		//put the URL together
		$url .= '?'.implode('&amp;',$vars);
		return $url;
	}
	
	/**
	 * Generates the base URL of a=cart without any other parameters.  Handy for instances where you need to
	 * create a url for a certain action, or to link to the cart, just append the extra parameters to the end
	 * of the url returned by this function.  It automatically accounts for the SSL setting in the 
	 * admin turned on or off, as long as $ssl is left to be true.  If $ssl is set to false, it will use 
	 * the non-ssl URL even if SSL is turned on in the admin.
	 *
	 * @param boolean $ssl
	 * @return string
	 */
	public function getCartBaseUrl ($ssl = true)
	{
		return $this->getProcessFormUrl($ssl, true);
	}
	
	/**
	 * Gets the action for the page.
	 * @return string
	 */
	public function getAction ()
	{
		return $this->action;
	}
	
	/**
	 * Gets the cost of all the items in the cart so far, by adding up all the getCost() values.  If
	 * $up_to_process_order is not 0 (default value), it will only add up items who's process 
	 * order is less than the process order specified.
	 *
	 * @param int $up_to_process_order
	 * @return float
	 */
	public function getCartTotal ($up_to_process_order = 0)
	{
		return $this->order->getOrderTotal($up_to_process_order);
	}
	
	private function _initReg ()
	{
		if (is_object($this->registry)) {
			return;
		}
		$this->registry = new geoRegistry;
		$this->registry->setName('cart');
		$this->registry->setId($this->cart_variables['id']);
		$this->registry->unSerialize();
	}
	
	/**
	 * Gets the user data and stores it in this->user_data in array format.
	 *
	 * @return unknown
	 */
	private function _getUserData($userId = null)
	{
		if (!defined('IN_ADMIN') && $userId === null) {
			$userId = $this->session->getUserId();
		}
		$anonUser = false;
		if (!$userId) {
			//return false;
			$anonReg = geoAddon::getRegistry('anonymous_listing');
			if($anonReg) {
				$userId = $anonReg->get('anon_user_id',0);
				$anonUser = true;
			}
		}
		$user = geoUser::getUser($userId);
		if (!$user) {
			//fallback to make sure user data is never empty handed, this will
			//happen whenever user is not logged in, and anon addon is not enabled
			$this->user_data = $this->db->GetRow("SELECT * FROM `geodesic_groups` WHERE `default_group` = 1 LIMIT 1");
			$this->user_data['id'] = 0;
			return true;
		}
		
		$this->user_data = $user->toArray();
		if ($anonUser) {
			//set user ID to 0
			$this->user_data['id'] = 0;
		}
		
		return true;
	}
	
	private $_price_plans = array();
	private $_default_price_plan;
	/**
	 * Gets the price plan for the user, and stores it in the cart variable
	 *
	 * @param int $pricePlanId
	 * @param int $category
	 * @return bool True if price plan was retrieved, false otherwise
	 */
	public function setPricePlan ($pricePlanId = 0, $category = 0)
	{
		$pricePlanId = intval($pricePlanId);
		$category = intval($category);
		
		if (!$pricePlanId || !geoPlanItem::isValidPricePlan($pricePlanId)) {
			if ($this->_default_price_plan) {
				$pricePlanId = $this->_default_price_plan;
			} else {
				//see if we can get the price plan id
				$pricePlanId = $this->_default_price_plan = ((geoPC::is_classifieds())? $this->user_data["price_plan_id"]: $this->user_data["auction_price_plan_id"]);
			}
			if (!geoPlanItem::isValidPricePlan($pricePlanId)) {
				//one we got is not valid!  Use default values as failsafe
				$pricePlanId = (geoPC::is_classifieds())? 1 : 5;
			}
		}
		
		if (!$pricePlanId) {
			//there was no price plan ID?
			//echo "no price plan id<Br>\n";
			return false;
		}
		
		//make sure the price plan is not already set...
		if (isset($this->_price_plans[$pricePlanId][$category])) {
			//price plan already retrieved before, don't need to keep getting it.
			$this->price_plan = $this->site->price_plan = $this->_price_plans[$pricePlanId][$category];
			//make sure the site class knows about the price plan change
			$this->site->users_price_plan = $this->price_plan['price_plan_id'];
			geoOrderItem::reorderTypes($pricePlanId,$category);
			return true;
		}
		//get price plan specifics
		
		$sql = "SELECT * FROM ".geoTables::price_plans_table." WHERE `price_plan_id` = ? LIMIT 1";
		$price_plan_result = $this->db->GetRow($sql, array($pricePlanId));
		if (!$price_plan_result) {
			trigger_error('ERROR CART SQL: Sql: '.$sql.' Error msg: '.$this->db->ErrorMsg());
			return false;
		}
		$this->price_plan = $price_plan_result;
		$original_category = $category;
		if ($category && $this->price_plan['type_of_billing'] == 1 && (geoPC::is_ent() || geoPC::is_premier())) {
			$stmt_cat_plan = $this->db->Prepare("SELECT * FROM ".geoTables::price_plans_categories_table." WHERE `price_plan_id` = ? AND `category_id` = ? LIMIT 1");
			$stmt_get_parent = $this->db->Prepare("SELECT `parent_id` FROM ".geoTables::categories_table." WHERE `category_id` = ? LIMIT 1");
			do {
				$show_price_plan = $this->db->GetRow($stmt_cat_plan, array($pricePlanId,$category));
				if ($show_price_plan === false) {
					trigger_error('ERROR CART SQL: Error msg: '.$this->db->ErrorMsg());
					return false;
				}
				
				if (count($show_price_plan) > 0) {
					//found the category price plan to use..
					break;
				}
				
				$show_price_plan = 0;
				
				//get category parent
				$show_category = $this->db->GetRow($stmt_get_parent, array($category));
				if ($show_category === false) {
					trigger_error('ERROR CART SQL: Sql: '.$sql.' Error msg: '.$this->db->ErrorMsg());
					return false;
				}
				
				if (isset($show_category['parent_id'])) {
					//parent category found
					$category = intval($show_category['parent_id']);
					continue;
				}
				trigger_error('DEBUG CART: Unable to get category price plan, category not found.');
				return false;
				
				//check all the way to the main category
			} while ($category != 0);
			
			if (isset($show_price_plan) && is_array($show_price_plan)) {
				//merge the category specific results with the price plan.
				$this->price_plan = array_merge($this->price_plan, $show_price_plan);
			}
		}
		//allow order items to add to the price plan
		//(price plans tied to order items, while user groups tied to payment gateways)
		geoOrderItem::callUpdate('geoCart_setPricePlan',array('price_plan_id'=>$pricePlanId,'category'=>$category));
		
		//cache it so we don't keep re-doing it for price plans/categories
		$this->_price_plans[$pricePlanId][$original_category] = $this->price_plan;
		//make sure the site knows about the price plan change
		$this->site->price_plan = $this->price_plan;
		$this->site->users_price_plan = $this->price_plan['price_plan_id'];
		
		geoOrderItem::reorderTypes($pricePlanId,$original_category);
		return true;
	}
	
	/**
	 * Gets an error message previously set (on same page load) using 
	 * geoCart::setErrorMsg()
	 *
	 * @param string $name
	 * @return string The error message, or an empty string if the error message is not found.
	 */
	public function getErrorMsg ($name)
	{
		if (isset($this->error_msgs[$name])) {
			return $this->error_msgs[$name];
		}
		return '';
	}
	
	/**
	 * Gets all the error messages in an associative array.
	 *
	 * @return array
	 */
	public function getErrorMsgs ()
	{
		return $this->error_msgs;
	}
	
	/**
	 * Whether or not the current cart is "in the middle of something", in other
	 * words, if you were to attempt to add something new to the cart, would it
	 * be interrupting something?  Be sure the cart is init before calling this,
	 * even if only items are inited.
	 * 
	 * @return bool True if it's in the middle of something, false otherwise.
	 * @since Version 4.1.0
	 */
	public function isInMiddleOfSomething ()
	{
		if (($this->isStandaloneCart() || ($this->main_type && $this->main_type != 'cart')) && $this->getAction() != 'cancel' && is_object($this->item)) {
			//it's in the middle of something all right!
			//note that if it is standalone cart, it's always in middle of something...
			return true;
		}
		return false;
	}
	
	private function _validateStep ($name)
	{
		if (is_array($name)) {
			//if array, go through each item and remove it from the array if it is not a valid step.
			foreach ($name as $k => $v) {
				if (!$this->_validateStep($v)) {
					unset($name[$k]);
				}
			}
			return $name;
		}
		if (in_array($name,$this->built_in_steps)) {
			return true;
		}
		$parts = explode(':',$name);
		if (count($parts) != 2) {
			//invalid!
			return false;
		}
		$item = geoOrderItem::getOrderItem($parts[0]);
		if (!is_object($item) || $item->getType() != $parts[0]) {
			//invalid!
			return false;
		}
		if (!method_exists($item,$parts[1].'CheckVars') || !method_exists($item, $parts[1].'Process') || !method_exists($item, $parts[1].'Display')) {
			//one of the required methods was not found, so invalid step,
			//all 3 are required for any step.
			trigger_error('DEBUG CART: eek!! required method not found!');
			return false;
		}
		unset($item);
		//got through all the checks, so must be a good step.
		return true;
	}
	
	/**
	 * Returns an array with the different parts of the step, like so:
	 * array(
	 * 	item_name => 'item_name',
	 * 	'step' => 'step'
	 * )
	 * 
	 * Assumes step has already been checked for validity.
	 *
	 * @param string $step_name
	 * @return array
	 */
	private function _getStepParts ($step)
	{
		if (in_array($step,$this->built_in_steps)) {
			//item is going to be the main item type
			$item_name = $this->main_type;
			$step_name = $step;
		} else {
			$parts = explode(':',$step);
			trigger_error('DEBUG CART: Parts exploded: <pre>'.print_r($parts,1).'</pre>');
			$item_name = $parts[0];
			$step_name = $parts[1];
		}
		$return = array(
			'item_name' => $item_name,
			'step' => $step_name
		);
		return ($return);
	}
	
	public function getCommonTemplateVars ()
	{
		$tpl_vars = array ();
		
		$tpl_vars['cart_url'] = $this->getCartBaseUrl();
		$tpl_vars['process_form_url'] = $this->getProcessFormUrl();
		$tpl_vars['in_admin'] = defined('IN_ADMIN');
		
		return $tpl_vars;
	}
}