<?php
//ListingFeed.class.php
/**
 * Holds the geoListingFeed class, which can render RSS and other types of things
 * based on a group of listings.
 * 
 * @package System
 * @since Version 5.0.0
 */
/**************************************************************************
Geodesic Classifieds & Auctions Platform 5.2
Copyright (c) 2001-2011 Geodesic Solutions, LLC
All rights reserved
http://geodesicsolutions.com
see license attached to distribution
**************************************************************************/
##########SVN Build Data##########
##                              ##
## This File's Revision:        ##
##  $Rev:: 20317              $ ##
## File last change date:       ##
##  $Date:: 2010-09-08 11:19:#$ ##
##                              ##
##################################

/**
 * Class that helps to render RSS or other types of feeds based on list of listings.
 * 
 * @package System
 * @since Version 5.0.0
 */
class geoListingFeed
{
	private $_settings = array();
	private $_whereClauses = array ();
	private $_selections = array ();
	private $_feedSql = '';
	private $_resultSet, $_feedType;
	
	/**
	 * Use this constant to specify the value can be set in the URL.
	 * @var string
	 */
	const URL_SET = 'set';
	
	/**
	 * Type of feed is RSS feed.
	 * @var string
	 */
	const RSS_FEED = 'rss';
	
	/**
	 * Type of feed is oodle feed.
	 * @var string
	 */
	const OODLE_FEED = 'oodle';
	
	/**
	 * Type of feed is generic (script defined)
	 * @var string
	 */
	const GENERIC_FEED = 'generic';
	
	/**
	 * BACKWARDS COMPATIBILITY ONLY, do not use
	 * @var string
	 */
	const OODLE_IMG_THUMBNAIL = 'thumb';
	
	/**
	 * BACKWARDS COMPATIBILITY ONLY, do not use
	 * @var unknown_type
	 */
	const OODLE_IMG_FULL = 'full';
	
	/**
	 * When using image_url, this is choice to use the thumb not the full image
	 * @var string
	 */
	const IMG_THUMB = 'thumb';
	
	/**
	 * When using image_url, this is choice to use the full not the thumb image
	 * @var string
	 */
	const IMG_FULL = 'full';
	
	const CAT_NAME = 'cat_name';
	
	const CAT_NAME_ID = 'cat_name_id';
	
	/**
	 * Set the type of feed based on one of the built-in feed types.
	 * 
	 * @param string $feedType Either {@link geoListingFeed::RSS_FEED} or {@link geoListingFeed::OODLE_FEED}
	 */
	public function setFeedType ($feedType)
	{
		$this->_feedType = $feedType;
		switch ($feedType) {
			case geoListingFeed::RSS_FEED:
				//RSS feed, set defaults
				
				if (!isset($this->tpl_file)) $this->tpl_file = 'rss_listings.tpl';
				if (!$this->maxListings || $this->maxListings <= 0) {
					//default to 20
					$this->maxListings = 20;
				}
				//make it get the image extras (like width, height, etc)
				$this->lead_image_extras = true;
				
				break;
				
			case geoListingFeed::OODLE_FEED:
				//OODLE feed, set defaults
				
				if (!isset($this->tpl_file)) $this->tpl_file = 'oodle_feed.tpl';
				if (!$this->maxListings || $this->maxListings <= 0) {
					//default to 20 or 500 if oodle
					$this->maxListings = 500;
				}
				//make it NOT get the image extras (like width, height, etc)
				$this->lead_image_extras = false;
				//clean description and get rid of HTML
				$this->clean_description = $this->clean_description_html = true;
				//turn on image URL
				$this->imageUrl = 1;
				
				break;
				
			case geoListingFeed::GENERIC_FEED:
				//break ommited on purpose
				
			default:
				//generic defaults
				
				if (!isset($this->tpl_file)) $this->tpl_file = 'rss_listings.tpl';
				if (!$this->maxListings || $this->maxListings <= 0) {
					//default to 20
					$this->maxListings = 20;
				}
				
				break;
		}
	}
	
	/**
	 * Add where clause to the SQL that gathers the listing data.
	 * @param string $whereClause
	 * @return geoListingFeed Returns self to allow chaining.
	 */
	public function addWhereClause ($whereClause)
	{
		if ($whereClause) {
			$this->_whereClauses[] = $whereClause;
		}
		//return this to allow chaining.
		return $this;
	}
	
	/**
	 * Adds a column (or array of columns) to be retrieved when getting listing
	 * data from the database.
	 * 
	 * @param string|array $selectColumn
	 */
	public function addSelectColumn ($selectColumn)
	{
		if (is_array($selectColumn)) {
			//passing in an array of em
			$this->_selections = array_merge($this->_selections, $selectColumn);
		} else {
			$this->_selections[] = $selectColumn;
		}
	}
	
	/**
	 * Generates the SQL query based on everything specified for what is to be
	 * retrieved.
	 * 
	 * @return string
	 */
	public function generateSql ()
	{
		if (!isset($this->_feedType)) {
			//make sure defaults are set
			$this->setFeedType(self::GENERIC_FEED);
		}
		require_once(CLASSES_DIR.'site_class.php');
		$site = Singleton::getInstance('geoSite');
		
		if ($this->catId == self::URL_SET) {
			if (isset($_GET['catId'])) {
				$this->catId = intval($_GET['catId']);
			} else {
				$this->catId = 0;
			}
		}
		if ($this->catId && $this->catId > 0) {
			$site->get_sql_in_statement(0, $this->catId);
			if (strlen(trim($site->in_statement)) == 0) {
				//No in statement for category, either it's invalid, or is wrong in DB
				$this->catId = 0;
			}
		} else {
			$this->catId = 0;
		}
		
		if ($this->catId && $site->in_statement) {
			//needs to be in category
			$this->addWhereClause("`category` ".$site->in_statement);
		}
		
		if ($this->userId == self::URL_SET && isset($_GET['userId'])) {
			$this->userId = intval($_GET['userId']);
		}
		$this->userId = (int)$this->userId;
		if ($this->userId > 0) {
			$this->addWhereClause("`seller` = $this->userId");
		}
		
		//do the type of listing
		if ($this->type == self::URL_SET && isset($_GET['type'])) {
			$this->type = trim($_GET['type']);
		}
		
		switch ($this->type) {
			case 'all_auction':
				if (!geoPC::is_auctions()) {
					//can't do this type
					break;
				}
				$this->addWhereClause("(`item_type` = 2 OR `item_type` = 4)");
				break;
			
			case 'reverse':
				if (!geoPC::is_auctions()) {
					//can't do this type
					break;
				}
				//special type for custom reverse auctions
				$this->addWhereClause("`item_type` = 4");
				break;
			
			case 'buy_now':
				if (!geoPC::is_auctions()) {
					//can't do this type
					break;
				}
				$this->addWhereClause("`item_type` = 2");
				$this->addWhereClause("`buy_now` > 0");
				break;
				
			case 'buy_now_only':
				if (!geoPC::is_auctions()) {
					//can't do this type
					break;
				}
				$this->addWhereClause("`item_type` = 2");
				$this->addWhereClause("`buy_now_only` = 1");
				break;
			
			case 'dutch':
				if (!geoPC::is_auctions()) {
					//can't do this type
					break;
				}
				$this->addWhereClause("`item_type` = 2");
				$this->addWhereClause("`auction_type` = 2");
				break;
			
			case 'classified':
				if (!geoPC::is_classifieds()) {
					//can't do this type
					break;
				}
				$this->addWhereClause("(`item_type` = 1 OR `item_type` = 3)");
				break;
			
			case 'all':
				//break ommited on purpose
			default:
				//no criteria to add to where clause
				break;
		}
		if (!isset($this->orderByClause)) {
			if ($this->orderBy == self::URL_SET && isset($_GET['orderBy'])) {
				$this->orderBy = trim($_GET['orderBy']);
			}
			$seed = rand(); //used for any of the featured ones
			switch ($this->orderBy) {
				case 'featured_1':
					$this->addWhereClause('`featured_ad` = 1');
					$this->orderByClause = "ORDER BY rand($seed)";
					break;
					
				case 'featured_2':
					$this->addWhereClause('`featured_ad_2` = 1');
					$this->orderByClause = "ORDER BY rand($seed)";
					break;
					
				case 'featured_3':
					$this->addWhereClause('`featured_ad_3` = 1');
					$this->orderByClause = "ORDER BY rand($seed)";
					
					break;
					
				case 'featured_4':
					$this->addWhereClause('`featured_ad_4` = 1');
					$this->orderByClause = "ORDER BY rand($seed)";
					
					break;
					
				case 'featured_5':
					$this->addWhereClause('`featured_ad_5` = 1');
					$this->orderByClause = "ORDER BY rand($seed)";
					
					break;
					
				case 'hottest':
					$this->orderByClause = 'ORDER BY `viewed` DESC';
					
					break;
					
				case 'expiring':
					$this->orderByClause = 'ORDER BY `ends` ASC';
					
					break;
					
				case 'old':
					//date asc
					$this->orderByClause = 'ORDER BY `date` ASC';
					
					break;
					
				case 'new':
					//break ommited on purpose
				default:
					//default to new listings
					//date desc
					$this->orderByClause = 'ORDER BY `date` DESC';
					
					break; 
			}
		}
		//only show live listings
		if (!$this->skipLive) {
			$this->addWhereClause("`live` = 1");
		}
		
		$feed->maxListings = (int)$feed->maxListings;
		
		//construct the query
		$this->addSelectColumn(array('`id`','`title`','`description`','`date`','`category`'));
		$fieldsDisplay = array();
		foreach ($this->show as $field => $useField) {
			if ($useField && !in_array("`$field`", $this->_selections)) {
				$this->addSelectColumn("`$field`");
				if ($field == 'price') {
					//get pre and post currency as well
					$this->addSelectColumn("`precurrency`");
					$this->addSelectColumn("`postcurrency`");
				}
			}
			if ($useField && !in_array($field, $fieldsDisplay)) {
				$fieldsDisplay[$field] = (isset($this->label[$field]))? $this->label[$field]: '';
			}
		}
		$this->fields = $fieldsDisplay;
		$join = '';
		if ($this->imageUrl || $this->leadImage || ($this->show['image'] && $this->imageCount == 1)) {
			$join = " c LEFT JOIN `geodesic_classifieds_images_urls` u 
				ON u.classified_id=c.id AND
				(u.display_order=1 OR u.display_order IS NULL)";
			$this->addSelectColumn('`u`.`thumb_url`');
			$this->addSelectColumn('`u`.`image_url`');
			if ($this->lead_image_extras) {
				$this->addSelectColumn('`u`.`image_width`');
				$this->addSelectColumn('`u`.`image_height`');
				$this->addSelectColumn('`u`.`image_text`');
				$this->addSelectColumn('`u`.`image_id`');
				$this->addSelectColumn('`u`.`display_order` image_display_order');
			}
		}
		$where = (count($this->_whereClauses))? 'WHERE '.implode(" AND ",$this->_whereClauses) : '';
		$this->_feedSql = "SELECT ".implode(", ",$this->_selections)." FROM `geodesic_classifieds` $join
			$where
			$this->orderByClause LIMIT $this->maxListings";
	}
	
	/**
	 * Process a listing's data and get additional info for the listing "on the fly",
	 * this is meant to be called by the actual template.
	 * 
	 * @param array $listing
	 */
	public function processListing ($params, $smarty)
	{
		$listing = $params['listing'];
		if (!$listing) {
			//nothing to do
			return;
		}
		
		//cache listing in geoListing class
		geoListing::addListingData($listing);
		
		$db = DataAccess::getInstance();
		$base = geoFilter::getBaseHref();
		
		if ($this->imageUrl) {
			$imageUrl = (($this->defaultImgType == self::IMG_THUMB || !$listing['image_url']) && $listing['thumb_url'])? $listing['thumb_url'] : $listing['image_url'];
			if ($imageUrl && strpos($imageUrl, 'http') !== 0) {
				//prepend it with base URL, the URL does not contain the
				//full URL
				$imageUrl = $base . $imageUrl;
			}
			$listing['imageUrl'] = $imageUrl;
		} else if (($this->show['image'] && $this->imageCount != -1) || $this->leadImage) {
			if ($this->show['image'] && $this->imageCount != 1) {
				//Get images for each listing, have to get more than 1 image for each listing.
				$imageCount = (int)$this->imageCount;
				$limit = ($imageCount)? " LIMIT $imageCount" : '';
				$stmt = $db->Prepare("SELECT * FROM `geodesic_classifieds_images_urls` WHERE `classified_id`=? ORDER BY `display_order`$limit");
			} else {
				$stmt = false;
			}
			
			if ($listing['image'] > 0 || $this->leadImage) {
				if ($stmt) {
					$rows = $db->Execute($stmt, array($listing['id']));
				} else if (isset($listing['image_id']) && $listing['image_id'] !== null) {
					//it's in listing data already
					$row = array (
						'thumb_url' => $listing['thumb_url'],
						'image_url' => $listing['image_url'],	
						'image_width' => $listing['image_width'],
						'image_height' => $listing['image_height'],
						'image_text' => $listing['image_text'],
						'display_order' => $listing['image_display_order'],
					);
					
					$rows = array ($row);
				} else {
					//listing with no images to show
					$rows = array();
				}
				
				$listing['images'] = null;
				foreach ($rows as $row) {
					//figure out the size
					$thumbUrl = $row['thumb_url'];
					if ($thumbUrl == 0) {
						//Thumbnail not created, use full image url
						$thumbUrl = $row['image_url'];
					}
					$dim = geoImage::getScaledSize((int)$row['image_width'], (int)$row['image_height'], $this->imageWidth, $this->imageHeight);
					$x = $dim['width'];
					$y = $dim['height'];
					
					//use "images" field, to preserve the # images in case
					//template wants to use that to display image count
					if (strpos($thumbUrl, 'http') !== 0) {
						//prepend it with base URL, the URL does not contain the
						//full URL
						$thumbUrl = $base . $thumbUrl;
					}
					$listing['images'][$row['display_order']] = array (
						'url' => $thumbUrl,
						'width' => $x,
						'height' => $y,
						'text' => $row['image_text']
					);
					
					if ($this->leadImage && $row['display_order'] == 1) {
						//set lead width and height
						if ($this->leadWidth != $this->imageWidth || $this->leadHeight != $this->imageHeight) {
							//figure out lead pic width and height
							$dim = geoImage::getScaledSize((int)$row['image_width'], (int)$row['image_height'], $this->leadWidth, $this->leadHeight);
							$x = $dim['width'];
							$y = $dim['height'];
						}
						$listing['images'][1]['lead_width'] = $x;
						$listing['images'][1]['lead_height'] = $y;
					}
				}
			}
		}
		
		if ($this->clean_all) {
			require_once CLASSES_DIR . 'order_items/_listing_placement_common.php';
			$varsToUpdate = _listing_placement_commonOrderItem::getListingVarsToUpdate();
			
			foreach ($listing as $key => $val) {
				if (is_numeric($key) || !isset($varsToUpdate[$key])) {
					//ignore
					continue;
				}
				
				switch ($varsToUpdate[$key]) {
					case 'toDB':
						if (is_array($val) && $key == 'seller_buyer_data' && geoPC::is_ent()) {
							//special case
							$val = unserialize($val);
						}
						$val = geoString::fromDB($val);
						break;
					case 'int':
						$val = intval($val);
						break;
					case 'float':
						$val = floatval($val);
						break;
					case 'bool':
						$val = (($val)? true: false);
						break;
					default:
						//not altered, for fields like "date"
						break;
				}
				if (array_key_exists($key,$translations)) {
					$key = $translations[$key];
				}
				$listing[$key] = $val;
			}
		}
		
		if ($this->clean_description) {
			//clean up description
			if (!$this->clean_all) $listing['description'] = geoString::fromDB($listing['description']);
			//convert br's to newlines
			if ($this->clean_description_html) {
				$listing['description'] = preg_replace('/<br[\s]*\/?>/i'," \n", $listing['description']);
				//get rid of tags
				$listing['description'] = geoFilter::replaceDisallowedHtml($listing['description'], true);
			}
			//get rid of ]]> which would end the cdata
			$listing['description'] = trim(str_replace(']]>', '', $listing['description']));
		}
		
		if ($this->show['price']) {
			//format the price
			$listing['price'] = geoString::displayPrice($listing['price'], $listing['precurrency'], $listing['postcurrency']);
		}
		if ($this->removeCurrencyColumns) {
			unset ($listing['precurrency'], $listing['postcurrency']);
		}
		
		if (isset($this->dateFormat)) {
			//convert date fields
			$dateFields = array ('date','ends','start_time','end_time');
			foreach ($dateFields as $dField) {
				if (isset($listing[$dField])) {
					$listing[$dField] = date($this->dateFormat, $listing[$dField]);
				}
			}
		}
		
		if (isset($listing['category']) && $this->categoryName) {
			$catInfo = geoCategory::getBasicInfo($listing['category']);
			$listing['category_name'] = $catInfo['category_name'];
		}
		
		if ($this->extraQuestions && isset($listing['id'])) {
			//Get the category questions
			$listing['questions'] = $db->GetAll("SELECT * FROM ".geoTables::classified_extra_table."
				WHERE `classified_id`=?", array((int)$listing['id']));
		}
		
		$smarty->assign('listing', $listing);
	}
	
	/**
	 * Generates the result set of listings which is stored internally.
	 */
	public function generateResultSet ()
	{
		$getListings = true;
		$db = DataAccess::getInstance();
		
		if (!defined('IN_ADMIN') && $db->get_site_setting('site_on_off')) {
			//get valid IP's
			if (!geoUtil::isAllowedIp()) {
				$getListings = false;
			}
		}
		
		if ($getListings) {
			//NOTE:  Using Execute allows result set to be iterated using foreach...
			$this->_resultSet = $db->Execute($this->_feedSql);
		} else {
			$this->_resultSet = array ();
			//use site off item to display
			$this->emptyItem = $this->siteOffItem;
		}
	}
	
	/**
	 * Magic method that renders the feed
	 */
	public function __toString ()
	{
		//set the content type to xml
		if (!$this->debug) header ("Content-Type: application/xml;");
		
		$tpl_type = (isset($this->tpl_type))? $this->tpl_type : geoTemplate::SYSTEM;
		$tpl_resource = (isset($this->tpl_resource))? $this->tpl_resource : 'ListingFeed';
		
		$tpl = new geoTemplate($tpl_type,$tpl_resource);
		
		//register smarty function to process each listing
		$tpl->register_function ('process_listing', array ($this, 'processListing'));
		
		$tpl->listings = $this->_resultSet;
		$tpl->assign($this->_settings);
		
		//set up filter for URL's
		$tpl->register_modifier('rewriteUrl',array('geoListingFeed', 'rewriteUrl'));
		
		$tpl->fetch($this->tpl_file, null, null, true);
		
		if ($this->debug) {
			//if debug, flush messages so they display
			trigger_error('FLUSH MESSAGES');
		}
	}
	
	/**
	 * Magic method that gets the given setting using the syntax $feed->setting
	 * @param string $name
	 * @return mixed
	 */
	public function __get ($name)
	{
		if (isset($this->_settings[$name])) {
			return $this->_settings[$name];
		}
		return null;
	}
	
	/**
	 * Magic method that sets the setting using the syntax $feed->setting = 'value'
	 */
	public function __set ($name, $value)
	{
		$this->_settings[$name] = $value;
	}
	
	/**
	 * Magic method that allows seeing if the given setting is set using syntax
	 * isset($feed->setting)
	 * 
	 * @param string $name
	 * @return bool
	 */
	public function __isset($name)
	{
		return isset($this->_settings[$name]);
	}
	
	/**
	 * Magic method that allows un-setting a setting using syntax unset($feed->setting)
	 * 
	 * @param string $name
	 */
	public function __unset($name)
	{
		unset($this->_settings[$name]);
	}
	
	static $_seoUtil = 'noset';
	
	/**
	 * Since listing feed can potentially be displaying listings in the thousands,
	 * it must be as efficient as possible.  This method is used to re-write URL's
	 * using SEO addon, but bypassing the normal methods to do so as those methods
	 * are not callibrated for this use so are not efficient enough for our needs.
	 * 
	 * @param string $string The URL to be re-written
	 * @return string The re-written URL (or the original URL if it should not be
	 *   re-written)
	 */
	public static function rewriteUrl ($string) {
		if (self::$_seoUtil === 'noset') {
			self::$_seoUtil = geoAddon::getUtil('SEO');
		}
		
		if (self::$_seoUtil) {
			return self::$_seoUtil->rewriteUrl($string);
		}
		return $string;
	}
}
