<?php
//Region.class.php
/**
 * Holds the geoRegion class.
 * 
 * @package System
 * @since Version 4.0.0
 */
/**************************************************************************
Geodesic Classifieds & Auctions Platform 5.2
Copyright (c) 2001-2011 Geodesic Solutions, LLC
All rights reserved
http://geodesicsolutions.com
see license attached to distribution
**************************************************************************/
##########SVN Build Data##########
##                              ##
## This File's Revision:        ##
##  $Rev:: 21352              $ ##
## File last change date:       ##
##  $Date:: 2011-05-04 12:11:#$ ##
##                              ##
##################################

/**
 * Stuff for regions and sub-regions.
 * 
 * @package System
 * @since Version 4.0.0
 */
class geoRegion {
	private $_messages = array();
	private static $_jsAdded = false;
	private $_countryNames = array();
	private $_errors = array();
	private $_regions = array(), $_subRegions = array(), $_propogatedRegions = array();
	private $_isFancy = null;
	
	
	public static $max_latitude,$min_latitude,$max_longitude,$min_longitude;
	/**
	 * Used by the search class.
	 *
	 * @param float $Latitude
	 * @param float $Longitude
	 * @param int $distance
	 */
	public static function RadiusAssistant ($Latitude, $Longitude, $distance, $units = geoNumber::UNITS_MILES)
	{		
		self::$max_latitude = geoNumber::lat2($Latitude, $Longitude, $distance, 0, $units);
		self::$min_latitude = geoNumber::lat2($Latitude, $Longitude, $distance, 180, $units);
		
		self::$max_longitude = geoNumber::long2($Latitude, $Longitude, $distance, 90, $units);
		self::$min_longitude = geoNumber::long2($Latitude, $Longitude, $distance, 270, $units);
		
		//check for pole crossings
		//if the search box crosses a pole, stop there
	
		if (round($Longitude - (geoNumber::long2($Latitude, $Longitude, $distance, 0, $units))) != 0 ) {
			//if long changes signs on a bearing of 0(north), we have crossed the north pole. set max lat to 90 (north pole)
			self::$max_latitude = 90;
		}
		if (round($Longitude - (geoNumber::long2($Latitude, $Longitude, $distance, 180, $units))) != 0 ) {
			//same thing, but in the other direction and for the south pole. set min lat to -90
			self::$min_latitude = -90;
		}
		
		return;
		
		//old way, not as accurate the further away from the equator it gets...
		/*
		$EQUATOR_LAT_MILE = 69.172; //distance in miles of one degree of latitude at the equator
		$EQUATOR_LAT_KM = 111.325; //the above measurement, in km
		$equatorLat = $EQUATOR_LAT_MILE; //use miles (might add a switch for this later)
		
		self::$max_latitude = $Latitude + $Miles / $equatorLat;
		self::$min_latitude = $Latitude - (self::$max_latitude - $Latitude);
		self::$max_longitude = $Longitude + $Miles / (cos(self::$min_latitude * M_PI / 180) * $equatorLat);
		self::$min_longitude = $Longitude - (self::$max_longitude - $Longitude);
		*/
	}
	
	/**
	 * Get instance of region class.
	 *
	 * @return geoRegion
	 */
	public static function getInstance ()
	{
		return Singleton::getInstance('geoRegion');
	}
	
	/**
	 * Whether or not the states will depend on what country is selected.
	 * 
	 * This caches the result,
	 * so it is save to call this function over and over w/o too much performance hit.
	 *
	 * @return bool
	 */
	public function isFancy ()
	{
		if ($this->_isFancy === null) {
			$db = DataAccess::getInstance();
			
			$assoc_sql = "SELECT count(DISTINCT parent_id) as total FROM ".geoTables::states_table." WHERE parent_id != 0";
			$row = $db->GetRow($assoc_sql);
			if( $row === false ){
				$this->_isFancy = false;
				trigger_error('ERROR SQL: sql: '.$assoc_sql.' Error Msg: '.$db->ErrorMsg());
				return false;
			}
			if ($row['total'] > 1) {
				$this->_isFancy = true;
			} else {
				$this->_isFancy = false;
			}
		}
		return $this->_isFancy;
	}
	
	/**
	 * Gets the number of sub-regions attached to the given region.
	 * 
	 * @param int $region_id
	 * @param string $checkColumn pass in name of column or empty string, if
	 *  name of column is passed in, will add "WHERE col_name = 1" to query.
	 * @return int
	 */
	public function getSubRegionCount ($region_id,$checkColumn = '')
	{
		$db = DataAccess::getInstance();
		$sql = "SELECT COUNT(*) amount FROM ".geoTables::states_table." WHERE parent_id=?";
		if ($checkColumn) {
			$sql .= " AND `$checkColumn`='1'";
		}
		$r = $db->GetRow($sql,array($region_id));
		if($r===false) {
			trigger_error('ERROR SQL: '.$sql.' Error Msg: '.$db->ErrorMsg());
			return 0;
		}
		return $r['amount'];
	}
	
	/**
	 * Gets the number of main regions.
	 * @param string $checkColumn pass in name of column or empty string, if
	 *  name of column is passed in, will add "WHERE col_name = 1" to query.
	 * @return int Number of regions found.
	 */
	public function getRegionCount ($checkColumn = '')
	{
		$db = DataAccess::getInstance();
		$sql = "SELECT COUNT(*) count FROM ".geoTables::countries_table;
		if($checkColumn) {
			$sql .= " WHERE `$checkColumn`='1'";
		}
		$row = $db->GetRow($sql);
		if($row===false) {
			trigger_error('ERROR SQL: '.$sql.' Error Msg: '.$db->ErrorMsg());
			return 0;
		}
		return $row['count'];
	}
	
	/**
	 * Gets the "name" for the given region ID.
	 * @param int $region_id
	 * @return string|bool will return false if problem occurs, or the name (empty
	 *  string if region id not found)
	 */
	public function getRegionNameById ($region_id)
	{
		$db = DataAccess::getInstance();
		$sql = "SELECT `name` FROM geodesic_countries WHERE country_id=? OR abbreviation=?";
		$r = $db->GetRow($sql,array($region_id,$region_id));
		if ($r===false) {
			trigger_error('ERROR SQL: '.$sql.' Error Msg: '.$db->ErrorMsg());
			return false;
		}
		return trim($r['name']);
	}
	
	/**
	 * Gets the name for the given region and sub region ID.
	 * @param int $s_id sub-region ID
	 * @param int $p_id region ID
	 * @return string|bool The name, or empty string if not found, or false if problem occured.
	 */
	public function getSubRegionNameById ($s_id,$p_id)
	{
		$db = DataAccess::getInstance();
		$sql = "SELECT name FROM ".geoTables::states_table." WHERE state_id=? AND parent_id=? LIMIT 1";
		$r = $db->GetRow($sql,array($s_id,$p_id));
		if($r===false) {
			trigger_error('ERROR SQL: '.$sql.' Error Msg: '.$db->ErrorMsg());
			return false;
		}
		
		if(isset($r['name'])) {
			return trim($r['name']);
		}
		return '';
	}
	
	
	/**
	 * get the abbreviation of a state supplying the state id and region id
	 *
	 * @param int $subregion_id
	 * @return string
	 */
	public function getSubRegionAbbreviationById ($subregion_id)
	{
		$db = DataAccess::getInstance();
		$sql = "SELECT abbreviation FROM ".geoTables::states_table." WHERE state_id=? LIMIT 1";
		$r = $db->GetRow($sql,array($subregion_id));
		if($r===false) {
			trigger_error('ERROR SQL: '.$sql.' Error Msg: '.$db->ErrorMsg());
			return '';
		}
		if (isset($r['abbreviation'])) {
			return trim($r['abbreviation']);
		}
		trigger_error("ERROR REGION STATS: invalid sub region");
		return '';
	}
	
	/**
	 * Notice: this functinality works only after/with  the country field.
	 * a country ID needs to be specified to state field and propagate the state field with the country specified  
	 *
	 * @param string $country_id
	 * @param array $countries
	 */
	private function _propagateStates ($country_id, $countries = array())
	{
		$db = DataAccess::getInstance();
		if (isset($this->_propogatedRegions[$country_id])) {
			return;
		}
		$sql = "SELECT * FROM ".geoTables::states_table." WHERE parent_id = ? ORDER BY `display_order`, `name`";
		$r = $db->GetAll($sql, array($country_id));
		$javascript = '';
		foreach ($r as $state) {
			//use addslashes to clean, since inserting as js.
			$abbr = trim(addslashes($state['abbreviation']));
			
			$parent = trim (addslashes($countries[$state['parent_id']]));
			$name = trim ( addslashes($state['name']));
			$javascript .= "addRegion(\"{$parent}\", \"{$abbr}\", \"{$name}\" );\n";
		}
		if($javascript) {
			self::_addJsFile();
			geoView::getInstance()->addTop("\n<script type='text/javascript'>\n$javascript\n</script>\n");
		}
		$this->_propogatedRegions[$country_id] = $country_id;
	}
	
	private static function _addJsFile ()
	{
		if (!self::$_jsAdded) {
			$location = geoTemplate::getUrl('js','regions.js');
			
			if (defined('IN_ADMIN')) {
				$location = '../'.$location;
			}
			
			geoView::getInstance()->addJScript($location);
			self::$_jsAdded = true;
		}
	}
	
	/**
	 * This is not implemented, I guess someone forgot to finish coding it, but
	 * I would guess that it does as the method name implies, if it were to be
	 * implemented.
	 * 
	 * @param int $region_id
	 * @return unknown_type
	 */
	public function getListingCountByRegion ($region_id)
	{
		
	}
	
	/**
	 * Get the number of listings in a particular sub region.
	 * @param int $subregion
	 * @param int $region
	 * @return int
	 */
	public function getListingCountBySubRegion ($subregion,$region)
	{
		if(is_numeric($subregion)) {
			$subregion = self::getSubRegionAbbreviationById($subregion,$region);
		}
		$db = DataAccess::getInstance();
		
		$sql = "SELECT COUNT(id) total FROM geodesic_classifieds WHERE location_state = ?";
		$r=$db->GetRow($sql, array($subregion));
		if($r===false) {
			trigger_error('ERROR SQL: '.$sql.' Error Msg: '.$db->ErrorMsg());
			return 0;
		}
		return $r['total'];
	}
	
	/**
	 * Gets a list of sub-regions, in array format good for using in a tempalte.
	 * @param int $region_id
	 * @param string $checkColumn pass in name of column or empty string, if
	 *  name of column is passed in, will add "WHERE col_name = 1" to query.
	 * @param bool $getAllFields If true, will return array with all fields from
	 *   the region table instead of just name and id.
	 * @return array|bool will return false if problem occured.
	 */
	public function getSubRegionList ($region_id, $checkColumn = '', $getAllFields = false)
	{
		if(!$region_id) {
			trigger_error("ERROR REGION: No region ID is specified or is invalid");
			return false;
		}
		$fields = ($getAllFields)? '*' : "state_id, name, parent_id, display_order display";
		$db = DataAccess::getInstance();
		$sql = "SELECT $fields FROM geodesic_states WHERE parent_id =?";
		$q_data = array($region_id);
		if($checkColumn) {
			$sql .= " AND `$checkColumn`='1'";
		}
		$sql .= " ORDER BY `display_order`, `name`";
		$r = $db->GetAll($sql, $q_data);
		if($r===false) {
			trigger_error('ERROR SQL: '.$sql.' Error Msg: '.$db->ErrorMsg());
			return false;
		}
		if(empty($r))  return false;
		$sub_regions = array();
		foreach ($r as $sub_region) {
			$sub_regions[$sub_region['state_id']] = $sub_region;
		}
		return $sub_regions;
	}
	
	/**
	 * get an array with all the regions available
	 * 
	 * @param string $checkColumn pass in name of column or empty string, if
	 *  name of column is passed in, will add "WHERE col_name = 1" to query.
	 * @param bool $getAllFields If true, will return array with all fields from
	 *   the region table instead of just name and id.
	 * @return Array
	 */
	public function getRegionsList ($checkColumn='', $getAllFields = false)
	{
		$db = DataAccess::getInstance();
		$fields = ($getAllFields)? '*' : "`country_id`,`name`";
		$sql = "SELECT $fields FROM `geodesic_countries`";// order by display_order, name";
		if($checkColumn) {
			$sql .= " WHERE `$checkColumn`='1'";
		}
		$sql .= " ORDER BY `display_order`,`name`";
		$r = $db->GetAssoc($sql);
		if($r===false) {
			trigger_error("Error gettings regions on geoRegion CLASS! ".$db->ErrorMsg());
			return false;
		}
		
		return $r;
	}
	
	/**
	 * Adds a region name,something to do with the javascript.
	 * 
	 * @param string $regionName
	 * @param string $subRegionName
	 */
	public function addRegionName ($regionName = 'c[country]', $subRegionName = 'c[state]') {
		if ($this->isFancy() && !in_array($regionName,$this->_countryNames)) {
			$this->_countryNames[$regionName] = $regionName;
			geoView::getInstance()->addTop("
			<script type='text/javascript'>
			regionNames[regionNames.length] = ['$regionName','$subRegionName'];
			</script>
			");
		}
	}
	/**
	 * Gets a bunch of HTML for display in a form.
	 * @param string $selected_region
	 * @param string $countryName The field name to use on the html tags.
	 * @param bool $useAbbreviations If true, will set the "value" to use abreviations, not the
	 *  country names.
	 * @return string
	 */
	public function getRegion ($selected_region='',$countryName = 'c[country]', $useAbbreviations = false)
	{
		$db = DataAccess::getInstance();
		$this->_messages = $db->get_text(true,15);
		$selected_region = trim($selected_region);
		$tpl = new geoTemplate('system', 'classes');
		
		//make sure at least the default is added
		$this->addRegionName();
			
		$sql = "SELECT * FROM geodesic_countries order by display_order, name";
		$r = $db->GetAll($sql);
		if ($r === false) {
			trigger_error("Error gettings regions on geoRegion CLASS! ".$db->ErrorMsg());
			return false;
		}
		
		$tpl->assign('fieldName', $countryName);
		
		if (count($r) == 1) {
			$value = ($useAbbreviations)? $r[0]['abbreviation']: $r[0]['name'];
			$one['value'] = $countries[$r[0]['country_id']] = trim(geoString::specialChars($value));
			$one['name'] = $r[0]['name'];
			$tpl->assign('one', $one);
			//only one country, display it and have a hidden field for it
			$this->_propagateStates($countries[$r[0]['country_id']],$countries);
		} else {
			//multiple countries, display drop down
			$options = array();
			$options[0]['value'] = 'none';
			$options[0]['label'] = $this->_messages[500075];
			$curr_dd_index = 1;
			$javascript = '';
			$countries = array();
			foreach($r as $show ){
				//spit out the country list
				$abbr = trim(geoString::specialChars($show['abbreviation']));
				$name = trim(geoString::specialChars($show['name']));
				if ($selected_region == $name || $selected_region == $abbr) {
					$options[$curr_dd_index]['selected'] = true;
				}
				//NOTE:  Value is escaped in the template.  It should not be escaped
				//in the JS, or it will not "match up" for countries with special chars
				$options[$curr_dd_index]['value'] = $countries[$show['country_id']] = ($useAbbreviations)? $show['abbreviation']: $show['name'];
				$options[$curr_dd_index]['label'] = trim($show['name']);
				$curr_dd_index++;
			}
			$tpl->assign('options', $options);
			
			if (isset($this->_errors[$countryName])) {
				$tpl->assign('error', $this->_messages[263]);
			}
			
			// Get country names for state drop down.
			foreach($r as $country) {
				$this->_propagateStates($country['country_id'], $countries);
			}
		}
		
		return $tpl->fetch('Region/get_region.tpl');
	}
	
	
	/**
	 * Gets a bunch of HTML to display the sub-region drop down in a form.
	 * 
	 * @param string $selected_subregion
	 * @param string $stateName The name that will be used in html tags.
	 * @return string
	 */
	public function getSubRegion ($selected_subregion = '',$stateName = 'c[state]')
	{
		$db = DataAccess::getInstance();
		$this->_messages = $db->get_text(true,15);
		$selected_subregion = trim($selected_subregion);
		// check for more than one country
		$fancy = $this->isFancy();
		// Will tell the software that javascript is needed, and if there are associations  -- no associations, no javascript.
		if( $fancy) {
			$sql = "SELECT * FROM ".geoTables::states_table." WHERE parent_id != 0 ORDER BY parent_id, display_order, name";
		} else {
			$sql = "SELECT * FROM ".geoTables::states_table." ORDER BY parent_id, display_order, name";
		}
			
		$states = $db->GetAll($sql);
		if (!$states) {
			trigger_error("DEBUG SQL: error getting states".$sql. ' '. $db->ErrorMsg());
			return false;
		}
		
		$tpl = new geoTemplate('system', 'classes');
		$tpl->stateName = $stateName;
		
		geoView::getInstance()->addTop("
			<script type='text/javascript'>
			if (!selectAStateText)
				var selectAStateText = '{$this->_messages[500074]}';
			</script>");
		$tpl->noneLabel = $this->_messages[500074];
		$parent = 0;
		$opt_group = false;
		
		$statesData = array();
		$tpl->fancy = $fancy;
		$tpl->parent = $parent;
		$tpl->opt_group = $opt_group;
		foreach ($states as $key => $state) {
			if($fancy) {
				if ( $parent != $state['parent_id']){
					$states[$key]['label'] = geoString::specialChars($countries[$state['parent_id']]);
					$opt_group = true;
					$parent = $state['parent_id'];
				}
			}
			
			$abbr = trim(geoString::specialChars($state['abbreviation']));
			$states[$key]['selected'] = ($abbr == $selected_subregion || $state['name'] == $selected_subregion) ? true : false;
			$states[$key]['value'] = $abbr;
			$states[$key]['text'] = $state['name'];
		}
		$tpl->states = $states;
		return $tpl->fetch('Region/get_subregion.tpl');
	}
}
